<?php

namespace yndenz\Modules\Banner;

use WP_Post;
use WP_Term;
use yndenz\Modules\BaseModule\BaseModule;

class Banner extends BaseModule {

	public $object = null;
	protected $_image = null;
	protected $_image_size = 'full';
	protected $_content = '';
	protected $_includes_title = false;

	/**
	 * Register the banner as custom field.
	 */
	public static function registerFields() {
		if ( ! function_exists( 'register_field_group' ) ) {
			return;
		}

		$locations = array(
			array(
				array(
					'param'    => 'post_type',
					'operator' => '==',
					'value'    => 'page',
					'order_no' => 0,
					'group_no' => 0,
				),
				array(
					'param'    => 'page_type',
					'operator' => '!=',
					'value'    => 'front_page',
					'order_no' => 1,
					'group_no' => 0,
				),
			),
			array(
				array(
					'param'    => 'post_type',
					'operator' => '==',
					'value'    => 'post',
					'order_no' => 0,
					'group_no' => 1,
				),
			),
			array(
				array(
					'param'    => 'ef_taxonomy',
					'operator' => '==',
					'value'    => 'category',
					'order_no' => 0,
					'group_no' => 2,
				),
			),
			array(
				array(
					'param'    => 'ef_taxonomy',
					'operator' => '==',
					'value'    => 'post_tag',
					'order_no' => 0,
					'group_no' => 3,
				),
			),
		);
		$locations = apply_filters( 'yndenz_modules_banner_locations', $locations );

		$fields = array(
			array(
				'key'          => 'field_banner_image',
				'label'        => _x( 'Banner afbeelding', 'Back-end add/edit page', 'base-theme' ),
				'name'         => 'banner_image',
				'type'         => 'image',
				'save_format'  => 'id',
				'preview_size' => 'medium',
				'library'      => 'all',
			),
			array(
				'key'           => 'field_banner_content',
				'label'         => _x( 'Banner content', 'Back-end add/edit page', 'base-theme' ),
				'name'          => 'banner_content',
				'type'          => 'textarea',
				'default_value' => '',
				'placeholder'   => '',
				'maxlength'     => '',
				'rows'          => '',
				'formatting'    => 'none',
			),
		);
		$fields = apply_filters( 'yndenz_modules_banner_fields', $fields );

		register_field_group( array(
			'id'         => 'group_banner',
			'title'      => _x( 'Banner', 'Back-end add/edit page', 'base-theme' ),
			'fields'     => $fields,
			'location'   => $locations,
			'options'    => array(
				'position'       => 'acf_after_title',
				'layout'         => 'default',
				'hide_on_screen' => array(),
			),
			'menu_order' => 0,
		) );
	}

	/**
	 * Get the scripts used for the banner.
	 *
	 * @return array
	 */
	public static function getScripts() {
		return array(
			'banner-script' => array( 'src' => get_modules_directory_uri() . 'banner/banner.min.js' )
		);
	}

	/**
	 * Display the banner.
	 *
	 * @param bool $parallax
	 */
	public function display( $parallax = false ) {
		if ( ! isset( $this->_image ) ) {
			$this->initialize();
		}

		$template = locate_template( 'modules/banner.php' );
		if ( empty( $template ) ) {
			$template = dirname( __FILE__ ) . '/templates/banner.php';
		}

		add_filter( 'base_theme_basics_scripts', array( static::class, 'addScripts' ) );

		include $template;
	}

	/**
	 * Initialize the banner details.
	 *
	 * @param string|null $title
	 * @param string      $image_size
	 */
	public function initialize( $title = null, $image_size = 'full' ) {
		$this->_image_size = $image_size;

		$this->setDefaults();
		$this->defineObject();
		$this->defineImage();
		$this->defineContent( $title );
	}

	protected function setDefaults() {
		if ( is_404() ) {
			$this->_image = of_get_option( '404_page_banner' );
		} else if ( is_main_query() && ! in_the_loop() ) {
			if ( is_search() ) {
				$this->_image   = of_get_option( 'search_banner' );
				$this->_content = get_search_form( false );
			} else if ( is_tag() ) {
				$this->_image   = of_get_option( 'default_tag_banner' );
				$content        = of_get_option( 'default_tag_banner_content' );
				$this->_content = preg_replace( '/<p class="heading-style">([^<]*)<\/p>/i', '<p class="heading-style">"' . single_tag_title( '', false ) . '"</p>', $content );
			}
		}
	}

	protected function defineObject() {
		global $post;
		$this->object = $post;

		if ( is_main_query() && ! in_the_loop() ) {
			if ( is_tag() ) {
				$this->object = get_tag( get_query_var( 'tag_id' ) );
			} else if ( is_archive() && is_category() ) {
				$this->object = get_category( get_query_var( 'cat' ) );
			}
		}

		if ( empty( $this->object ) ) {
			$this->object = get_queried_object();
		}
	}

	protected function defineImage() {
		if ( is_object( $this->object ) ) {
			if ( $this->object instanceof WP_Post ) {
				$custom_image = get_post_meta( $this->object->ID, 'banner_image', true );
			} else if ( $this->object instanceof WP_Term ) {
				$custom_image = get_field( 'banner_image', "{$this->object->taxonomy}_{$this->object->term_id}" );
			}

			if ( ! empty( $custom_image ) ) {
				$this->_image = $custom_image;
			}
		}

		if ( preg_match( '/^[0-9]+$/i', $this->_image ) ) {
			$this->_image = (int) $this->_image;
		}

		if ( ! empty( $this->_image ) && ! is_numeric( $this->_image ) ) {
			$this->_image = get_attachment_id_from_url( $this->_image );
		}

		if ( empty( $this->_image ) && defined( 'DEFAULT_BANNER_IMAGE' ) ) {
			$this->_image = DEFAULT_BANNER_IMAGE;
		}

		$this->_image = apply_filters( 'yndenz_modules_banner_image', $this->_image );
	}

	protected function defineContent( $title = null ) {
		if ( is_object( $this->object ) ) {
			if ( $this->object instanceof WP_Post ) {
				$custom_content = get_post_meta( $this->object->ID, 'banner_content', true );
			} else if ( $this->object instanceof WP_Term ) {
				$custom_content = get_field( 'banner_content', "{$this->object->taxonomy}_{$this->object->term_id}", false );
			}

			if ( ! empty( $custom_content ) ) {
				$this->_content = $custom_content;
			}
		}

		$this->_includes_title = preg_match( '~<p class="heading-style">(.*)</p>~is', $this->_content ) > 0;
		if ( ! is_null( $title ) ) {
			if ( $this->_includes_title ) {
				$this->_content = preg_replace( '~<p class="heading-style">(.*)</p>~is', '<p class="heading-style">' . $title . '</p>', $this->_content );
			} else {
				$this->_content = '<p class="heading-style">' . $title . '</p>' . $this->_content;
			}
			$this->_includes_title = true;
		}

		$this->_content = apply_filters( 'yndenz_modules_banner_content', $this->_content );
	}

	/**
	 * Display the banner image.
	 */
	public function displayImage() {
		if ( empty( $this->_image ) ) {
			return;
		}

		$template = locate_template( 'modules/banner-image.php' );
		if ( empty( $template ) ) {
			$template = dirname( __FILE__ ) . '/templates/banner-image.php';
		}

		include $template;
	}

	/**
	 * Check if the title is included in the banner.
	 *
	 * @return bool
	 */
	public function isTitleIncluded() {
		return $this->_includes_title;
	}

	/**
	 * Get the banner image attachment ID.
	 *
	 * @return int|null
	 */
	public function getImage() {
		return $this->_image;
	}

	/**
	 * Get the banner image size.
	 *
	 * @return string
	 */
	public function getImageSize() {
		return $this->_image_size;
	}

	/**
	 * Check if the banner has content.
	 *
	 * @return bool
	 */
	public function hasContent() {
		return ! empty( $this->_content );
	}

	/**
	 * Get the banner content.
	 *
	 * @return string
	 */
	public function getContent() {
		return $this->_content;
	}

}