<?php

namespace yndenz\Modules\Employee;

use WP_Query;
use yndenz\Modules\BaseModule\BaseModule;

class Employee extends BaseModule {

    protected $initialized = false;

    /** @var WP_Query */
    protected $query;

    public static function register() {
        static::registerPostType();

        add_shortcode('specialist', array(static::class, 'shortcode'));
        add_shortcode('list_employees', array(static::class, 'shortcodeList'));
        add_shortcode('employees', array(static::class, 'shortcodeList'));
    }

    public static function registerPostType() {
        $args = array(
            'labels' => array(
                'name' => _x('Medewerkers', 'Back-end menu', 'base-theme'),
                'singular_name' => _x('medewerker', 'Back-end medewerker toevoegen/bewerken', 'base-theme'),
                'add_new_item' => _x('Medewerker toevoegen', 'Back-end medewerker toevoegen', 'base-theme'),
                'edit_item' => _x('Medewerker bewerken', 'Back-end medewerker bewerken', 'base-theme'),
            ),
            'description' => _x('Medewerkers', 'Back-end medewerker toevoegen/bewerken', 'base-theme'),
            'public' => false,
            'publicly_queryable' => false,
            'show_ui' => true,
            'show_in_menu' => true,
            'menu_position' => 20,
            'menu_icon' => 'dashicons-businessman',
            'capability_type' => 'post',
            'has_archive' => false,
            'hierarchical' => false,
            'supports' => array('title', 'editor', 'thumbnail')
        );

        $args = apply_filters('yndenz_modules_employee_post_type_args', $args);

        register_post_type('employee', $args);
    }

    public static function registerFields() {
        if (!function_exists("register_field_group")) {
            return;
        }

        static::registerContactInformation();
        static::registerSpecialist();
    }

    public static function registerContactInformation() {
        $fields = array(
            array(
                'key' => 'field_employee_role',
                'label' => _x('Rol', 'Back-end medewerker toevoegen/bewerken', 'base-theme'),
                'name' => 'employee_role',
                'type' => 'text',
                'default_value' => '',
                'placeholder' => '',
                'prepend' => '',
                'append' => '',
                'formatting' => 'html',
                'maxlength' => '',
            ),
            array(
                'key' => 'field_employee_email',
                'label' => _x('E-mailadres', 'Back-end medewerker toevoegen/bewerken', 'base-theme'),
                'name' => 'employee_email',
                'type' => 'text',
                'default_value' => '',
                'placeholder' => '',
                'prepend' => '',
                'append' => '',
                'formatting' => 'none',
                'maxlength' => '',
            ),
            array(
                'key' => 'field_employee_phone',
                'label' => _x('Telefoonnummer', 'Back-end medewerker toevoegen/bewerken', 'base-theme'),
                'name' => 'employee_phone',
                'type' => 'text',
                'default_value' => '',
                'placeholder' => '',
                'prepend' => '',
                'append' => '',
                'formatting' => 'none',
                'maxlength' => '',
            ),
        );

        $fields = apply_filters('yndenz_modules_employee_contact_fields', $fields);

        register_field_group(array(
            'id' => 'acf_employee-contact-information',
            'title' => _x('Contactinformatie', 'Back-end medewerker toevoegen/bewerken', 'base-theme'),
            'fields' => $fields,
            'location' => array(
                array(
                    array(
                        'param' => 'post_type',
                        'operator' => '==',
                        'value' => 'employee',
                        'order_no' => 0,
                        'group_no' => 0,
                    ),
                ),
            ),
            'options' => array(
                'position' => 'acf_after_title',
                'layout' => 'default',
                'hide_on_screen' => array(),
            ),
            'menu_order' => 0,
        ));
    }

    public static function registerSpecialist() {
        $locations = array(
            array(
                array(
                    'param' => 'post_type',
                    'operator' => '==',
                    'value' => 'post',
                    'order_no' => 0,
                    'group_no' => 0,
                ),
            ),
            array(
                array(
                    'param' => 'post_type',
                    'operator' => '==',
                    'value' => 'page',
                    'order_no' => 0,
                    'group_no' => 1,
                ),
                array(
                    'param' => 'page_type',
                    'operator' => '!=',
                    'value' => 'front_page',
                    'order_no' => 1,
                    'group_no' => 1,
                ),
            ),
        );

        $locations = apply_filters('yndenz_modules_employee_specialist_locations', $locations);

        register_field_group(array(
            'id' => 'acf_specialists',
            'title' => _x('Specialisten', 'Back-end pagina toevoegen/bewerken', 'base-theme'),
            'fields' => array(
                array(
                    'key' => 'field_specialists',
                    'label' => _x('Medewerkers', 'Back-end pagina toevoegen/bewerken', 'base-theme'),
                    'name' => 'specialists',
                    'type' => 'post_object',
                    'instructions' => _x('Selecteer één of meerdere medewerkers. Als er meerdere medewerkers geselecteerd zijn, wordt er elke keer dat iemand de pagina bezoekt willekeurig één medewerker getoond.', 'Back-end pagina toevoegen/bewerken', 'base-theme'),
                    'post_type' => array(
                        0 => 'employee',
                    ),
                    'taxonomy' => array(
                        0 => 'all',
                    ),
                    'allow_null' => 1,
                    'multiple' => 1,
                ),
            ),
            'location' => $locations,
            'options' => array(
                'position' => 'side',
                'layout' => 'default',
                'hide_on_screen' => array(),
            ),
            'menu_order' => 0,
        ));
    }

    /**
     * Display a single employee.
     *
     * @param array|string $atts List of attributes added to shortcode.
     *
     * @return string
     */
    public static function shortcode($atts = array()) {
        ob_start();
        (new static)->display($atts);
        return ob_get_clean();
    }

    /**
     * Display a list of employees.
     *
     * @param array|string $atts List of attributes added to shortcode.
     *
     * @return string
     */
    public static function shortcodeList($atts = array()) {
        ob_start();
        (new static)->displayList($atts);
        return ob_get_clean();
    }

    /**
     * Fetch a list of employees.
     *
     * @param array $args
     */
    public function initialize($args = array()) {
        $args = wp_parse_args($args, array(
            'post_type' => 'employee',
            'posts_per_page' => -1,
        ));

        $args = apply_filters('yndenz_modules_employee_query_args', $args);

        $this->query = new WP_Query($args);

        $this->initialized = true;
    }

    /**
     * Display a single employee.
     *
     * @param array|string $args List of attributes added to shortcode.
     */
    public function display($args = array()) {
        if (!is_array($args)) {
            $args = shortcode_parse_atts($args);
        }

        $employee_ids = static::getSpecialistIds($args);
        if (empty($employee_ids)) {
            return;
        }

        $this->initialize(array('posts_per_page' => 1, 'orderby' => 'rand', 'post__in' => $employee_ids));

        if (!$this->query->have_posts()) {
            return;
        }

        $employee = $this->query->posts[0];
        include static::getSpecialistTemplate($args);
    }

    /**
     * Get list of related specialist IDs.
     *
     * @param array $args
     * @return array
     */
    public static function getSpecialistIds($args = array()) {
        if (!empty($args) && array_key_exists('id', $args)) {
            $employee_ids = array($args['id']);
        } else {
            $employee_ids = get_field('specialists');
        }

        if (is_array($employee_ids)) {
            $employee_ids = array_map(function ($employee_id) {
                return is_object($employee_id) ? $employee_id->ID : intval($employee_id);
            }, $employee_ids);
        }

        return apply_filters('yndenz_modules_employee_specialists', $employee_ids);
    }

    /**
     * Get template file to be used to display specialist.
     *
     * @param array $args
     * @return string
     */
    public static function getSpecialistTemplate($args = array()) {
        if (!empty($args) && array_key_exists('template', $args) && !empty($args['template'])) {
            $template = $args['template'];
        } else {
            $template_name = 'specialist';
            if (!empty($args) && array_key_exists('author', $args) && boolval($args['author'])) {
                $template_name .= '-author';
            }

            $template = locate_template("modules/{$template_name}.php");
            if (empty($template)) {
                $template = dirname(__FILE__) . "/templates/{$template_name}.php";
            }
        }
        return $template;
    }

    /**
     * Display a listing of employees.
     *
     * @param array|string $atts List of attributes added to shortcode.
     * @param string $content Content in between shortcode tags.
     */
    public function displayList($atts = array(), $content = '') {
        if (!is_array($atts)) {
            $atts = shortcode_parse_atts($atts);
        }

        if (!$this->initialized) {
            $args = $atts;
            if (is_array($atts) && array_key_exists('ids', $atts)) {
                $args['post__in'] = explode(',', $atts['ids']);
                unset($args['ids']);
            }
            $this->initialize($args);
        }

        if (!$this->query->have_posts()) {
            return;
        }

        if (!empty($atts) && array_key_exists('template', $atts)) {
            $template = $atts['template'];
        }
        if (empty($template)) {
            $template = locate_template('modules/specialist-list.php');
        }
        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/specialist-list.php';
        }

        include $template;
    }

}