<?php

namespace yndenz\Modules\JSON_LD;

use yndenz\Modules\BaseModule\BaseModule;

class JSON_LD extends BaseModule {

	private $_data = array();

	public static function register() {
		if (defined('WPSEO_VERSION') && version_compare(WPSEO_VERSION, '11.0', '<')) {
			add_action( 'wpseo_head', array( static::class, 'setup' ), 89 );
		}
	}

	public static function setup() {
		$instance = new static;
		$instance->initialize();

		// Filter out organisation and breadcrumb to add to WebPage or Article
		add_filter( 'wpseo_json_ld_output', array( $instance, 'filterFromYoastSEO' ), 999, 2 );

		// Output custom JSON+LD
		add_action( 'wpseo_json_ld', array( $instance, 'output' ), 99 );
	}

	/**
	 * Schema for WebPage or Article.
	 * See https://schema.org/WebPage
	 */
	public function initialize() {
		$type = $this->getType();

		$type = apply_filters( 'yndenz_modules_json_ld_page_type', $type );

		$this->_data['page'] = array(
			'@context' => 'https://schema.org',
			'@type'    => $type
		);

		if ( is_front_page() ) {
			$this->_data['page']['url']  = home_url( '/' );
			$this->_data['page']['name'] = get_bloginfo( 'name' );
		}

		if ( is_single() ) {
			$this->_data['page']['datePublished'] = get_post_time( 'c' );
			$this->_data['page']['dateModified']  = get_post_modified_time( 'c' );
		}

		if ( $this->_data['page']['@type'] === 'Article' ) {
			$this->_data['page']['headline'] = get_the_title();
		}
	}

	public function getType() {
		if ( is_front_page() ) {
			return 'WebSite';
		}
		if ( is_search() ) {
			return 'SearchResultsPage';
		}
		if ( is_archive() ) {
			return 'CollectionPage';
		}
		if ( is_singular( 'post' ) ) {
			return 'Article';
		}
		if ( stripos( get_permalink(), '/contact/' ) !== false ) {
			return 'ContactPage';
		}

		return 'WebPage';
	}

	/**
	 * Filter organisation and breadcrumb schema out of Yoast SEO JSON+LD to use them as property of another schema.
	 *
	 * @param string $context
	 * @param array  $data
	 *
	 * @return array
	 */
	public function filterFromYoastSEO( $data, $context ) {
		if ( empty( $data ) ) {
			return $data;
		}

		unset( $data['@context'] );

		if ( $context === 'company' ) {
			if ( array_key_exists( 'logo', $data ) ) {
				$data['logo'] = array(
					'@type'      => 'ImageObject',
					'contentUrl' => $data['logo'],
					'url'        => $data['logo']
				);
			}

			$this->_data['page']['publisher'] = $data;

			if ( is_singular( 'post' ) ) {
				$this->_data['page']['author'] = $data;
			}
		} else if ( $context === 'breadcrumb' && $this->_data['page']['@type'] !== 'Article' ) {
			$this->_data['page']['breadcrumb'] = $data;
		}

		return array();
	}

	/**
	 * Outputs the JSON LD code in a valid JSON+LD wrapper.
	 */
	public function output() {
		$this->_data = apply_filters( 'yndenz_modules_json_ld_data', $this->_data );

		foreach ( $this->_data as $context => $data ) {
			$data = apply_filters( 'yndenz_modules_json_ld_output', $data, $context );

			if ( is_array( $data ) && ! empty( $data ) ) {
				echo "<script type='application/ld+json'>", wp_json_encode( $data ), '</script>', "\n";
			}
		}
	}

}
