<?php

namespace yndenz\Modules\Reviews;

use stdClass;
use WP_Query;
use yndenz\Modules\BaseModule\BaseModule;

class Reviews extends BaseModule {

    public $query, $result;
    protected $initialized = false;

    /**
     * Register the review custom post type.
     */
    public static function register() {
        $args = array(
            'labels' => array(
                'name' => __('Recensie', 'base-theme'),
                'singular_name' => __('Recensie', 'base-theme'),
                'menu_name' => __('Recensies', 'base-theme'),
                'name_admin_bar' => __('Recensie', 'base-theme'),
                'add_new' => __('Toevoegen', 'base-theme'),
                'add_new_item' => __('Recensie toevoegen', 'base-theme'),
                'new_item' => __('Nieuwe recensie', 'base-theme'),
                'edit_item' => __('Bewerken', 'base-theme'),
                'view_item' => __('Bekijken', 'base-theme'),
                'all_items' => __('Alle recensies', 'base-theme'),
                'search_items' => __('Recensies zoeken', 'base-theme'),
                'not_found' => __('Geen recensies gevonden.', 'base-theme'),
                'not_found_in_trash' => __('Geen recensies gevonden in de prullenbak.', 'base-theme'),
            ),
            'description' => __('Recensies', 'base-theme'),
            'public' => false,
            'publicly_queryable' => false,
            'show_ui' => true,
            'show_in_menu' => true,
            'menu_position' => 20,
            'menu_icon' => 'dashicons-testimonial',
            'capability_type' => 'post',
            'has_archive' => false,
            'hierarchical' => false,
            'supports' => array('title', 'editor', 'thumbnail')
        );

        $args = apply_filters('yndenz_modules_reviews_post_type_args', $args);

        register_post_type('review', $args);
    }

    public static function registerFields() {
        if (!function_exists("register_field_group")) {
            return;
        }

        $fields = array(
            array(
                'key' => 'acf-review_role',
                'label' => __('Rol', 'base-theme'),
                'name' => 'review_role',
                'type' => 'text',
                'default_value' => '',
                'placeholder' => '',
                'prepend' => '',
                'append' => '',
                'formatting' => 'html',
                'maxlength' => '',
            ),
            array(
                'key' => 'acf-review_rating',
                'label' => __('Cijfer', 'base-theme'),
                'name' => 'review_rating',
                'type' => 'text',
                'default_value' => '',
                'placeholder' => '',
                'prepend' => '',
                'append' => '',
                'formatting' => 'none',
                'maxlength' => '',
            ),
        );

        $fields = apply_filters('yndenz_modules_reviews_fields', $fields);

        register_field_group(array(
            'id' => 'acf_review-contact-information',
            'title' => __('Contactinformatie', 'base-theme'),
            'fields' => $fields,
            'location' => array(
                array(
                    array(
                        'param' => 'post_type',
                        'operator' => '==',
                        'value' => 'review',
                        'order_no' => 0,
                        'group_no' => 0,
                    ),
                ),
            ),
            'options' => array(
                'position' => 'acf_after_title',
                'layout' => 'default',
                'hide_on_screen' => array(),
            ),
            'menu_order' => 0,
        ));
    }

    public function initialize($args = array(), $num_items = null, $pagination = false) {
        $args = wp_parse_args($args, array('post_type' => 'review', 'posts_per_page' => -1));
        $this->query = new WP_Query($args);
        if (!$this->query->have_posts()) {
            $this->initialized = true;
            return;
        }

        $result = new stdClass();
        $result->num_reviews = $this->query->found_posts;

        $reviews = array();
        foreach ($this->query->posts as $review) {
            $item = new stdClass();
            $item->author = get_the_title($review);
            $item->date = get_the_date('Y-m-d', $review);
            $item->rating = (float)str_replace(',', '.', get_post_meta($review->ID, 'review_rating', true));
            $item->reason = get_post_meta($review->ID, 'review_role', true);
            $item->text = $review->post_content;
            array_push($reviews, $item);
        }

        $result->average = array_sum(array_map(function ($item) {
            return $item->rating;
        }, $reviews)) / $result->num_reviews;

        if (is_null($num_items)) {
            $num_items = get_option('posts_per_page');
        }
        $offset = 0;
        if ($pagination) {
            $page = get_query_var('paged', 1);
            $offset = ($page - 1) * $num_items;
        }
        $result->reviews = array_slice($reviews, $offset, $num_items);

        $result->item_reviewed = static::itemReviewed();

        $this->result = $result;
        $this->initialized = true;
    }

    /**
     * Get the item reviewed.
     *
     * @return string
     */
    public static function itemReviewed() {
        $template = locate_template('modules/reviews-item-reviewed.php');

        if (empty($template)) {
            $template = locate_template('klantenvertellen-item-reviewed.php');
        }

        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/reviews-item-reviewed.php';
        }

        ob_start();
        include $template;
        return ob_get_clean();
    }

    /**
     * Display a listing of reviews.
     */
    public function display() {
        if (!$this->initialized) {
            $this->initialize(array(), null, true);
        }

        if (!isset($this->result)) {
            return;
        }

        $template = locate_template('modules/reviews-list.php');

        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/reviews-list.php';
        }

        include $template;
    }

    /**
     * Display a listing of reviews.
     */
    public function displayAverage() {
        $template = locate_template('modules/reviews-average.php');

        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/reviews-average.php';
        }

        include $template;
    }

    /**
     * Display a gallery with reviews.
     *
     * @param int $items_per_slide
     */
    public function displayGallery($items_per_slide = 1) {
        if (!$this->initialized) {
            $num_items = get_option('posts_per_page');
            if ($num_items % $items_per_slide > 0) {
                $num_items = $num_items + ($num_items % $items_per_slide);
            }
            $this->initialize(array(), $num_items);
        }

        if (!isset($this->result)) {
            return;
        }

        $template = locate_template('modules/reviews-gallery.php');

        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/reviews-gallery.php';
        }

        include $template;
    }

    /**
     * Display a single review.
     *
     * @param stdClass $review
     * @param bool $full_content
     */
    public function displayReview($review, $full_content = false, $show_read_more = true) {
        $template = locate_template('modules/reviews-review.php');

        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/reviews-review.php';
        }

        include $template;
    }

}