<?php

namespace yndenz\Modules\Reviews;

use stdClass;
use WP_Query;
use yndenz\Modules\BaseModule\BaseModule;

class Reviews extends BaseModule {

    public $query, $result;
    protected $initialized = false;

    /**
     * Register the review custom post type.
     */
    public static function register() {
	    static::registerPostType();

		if (defined('WPSEO_VERSION') && version_compare(WPSEO_VERSION, '11.0', '<')) {
	        add_filter('yndenz_modules_json_ld_data', array(static::class, 'JSON_LD'));
		} else {
			add_filter('wpseo_schema_article', array(static::class, 'JSON_LD'));
		}

        add_shortcode('reviews', array(static::class, 'shortcode'));
        add_shortcode('reviews-average', array(static::class, 'shortcodeAverage'));
        add_shortcode('reviews-gallery', array(static::class, 'shortcodeGallery'));
    }

    public static function registerPostType() {
        $args = array(
            'labels' => array(
                'name' => _x('Recensies', 'Breadcrumb / Back-end menu', 'base-theme'),
                'singular_name' => _x('recensie', 'Back-end recensie toevoegen/bewerken', 'base-theme'),
                'add_new_item' => _x('Recensie toevoegen', 'Back-end recensie toevoegen', 'base-theme'),
                'edit_item' => _x('Recensie bewerken', 'Back-end recensie bewerken', 'base-theme'),
            ),
            'description' => _x('Recensies', 'Back-end recensie toevoegen/bewerken', 'base-theme'),
            'public' => false,
            'publicly_queryable' => false,
            'show_ui' => true,
            'show_in_menu' => true,
            'menu_position' => 20,
            'menu_icon' => 'dashicons-testimonial',
            'capability_type' => 'post',
            'has_archive' => false,
            'hierarchical' => false,
            'supports' => array('title', 'editor', 'thumbnail')
        );

        $args = apply_filters('yndenz_modules_reviews_post_type_args', $args);

        register_post_type('review', $args);
    }

    public static function registerFields() {
        if (!function_exists("register_field_group")) {
            return;
        }

        static::registerContactDetails();
    }

    public static function registerContactDetails() {
        $fields = array(
            array(
                'key' => 'field_review_role',
                'label' => _x('Rol', 'Back-end recensie toevoegen/bewerken', 'base-theme'),
                'name' => 'review_role',
                'type' => 'text',
                'default_value' => '',
                'placeholder' => '',
                'prepend' => '',
                'append' => '',
                'formatting' => 'html',
                'maxlength' => '',
            ),
            array(
                'key' => 'field_review_rating',
                'label' => _x('Cijfer', 'Back-end recensie toevoegen/bewerken', 'base-theme'),
                'name' => 'review_rating',
                'type' => 'text',
                'default_value' => '',
                'placeholder' => '',
                'prepend' => '',
                'append' => '',
                'formatting' => 'none',
                'maxlength' => '',
            ),
        );

        $fields = apply_filters('yndenz_modules_reviews_fields', $fields);

        register_field_group(array(
            'id' => 'acf_review-contact-information',
            'title' => _x('Contactinformatie', 'Back-end recensie toevoegen/bewerken', 'base-theme'),
            'fields' => $fields,
            'location' => array(
                array(
                    array(
                        'param' => 'post_type',
                        'operator' => '==',
                        'value' => 'review',
                        'order_no' => 0,
                        'group_no' => 0,
                    ),
                ),
            ),
            'options' => array(
                'position' => 'acf_after_title',
                'layout' => 'default',
                'hide_on_screen' => array(),
            ),
            'menu_order' => 0,
        ));
    }

	public static function JSON_LD( $data ) {
		$reviews = new static;
		$reviews->initialize();

		if ( ! isset( $reviews->result ) ) {
			return $data;
		}

		$aggregate_rating = array(
			'@type'       => 'AggregateRating',
			'reviewCount' => $reviews->result->num_reviews,
			'bestRating'  => 10,
			'ratingValue' => round( $reviews->result->average, 1 )
		);

		if ( array_key_exists( 'publisher', $data['page'] ) ) {
			$aggregate_rating['itemReviewed'] = $data['page']['publisher'];
		}

		$aggregate_rating = apply_filters( 'yndenz_modules_reviews_json_ld_aggregate_rating', $aggregate_rating );

		if ( empty( $aggregate_rating ) || ! array_key_exists( 'itemReviewed', $data ) ) {
			return $data;
		}

		$data['aggregateRating'] = $aggregate_rating;

		return $data;
	}

    /**
     * Display a list of reviews.
     *
     * @param array|string $atts List of attributes added to shortcode.
     *
     * @return string
     */
    public static function shortcode($atts = array()) {
        ob_start();
        (new static)->displayList($atts);
        return ob_get_clean();
    }

    /**
     * Display the reviews average rating.
     *
     * @param array|string $atts List of attributes added to shortcode.
     *
     * @return string
     */
    public static function shortcodeAverage($atts = array()) {
        ob_start();
        (new static)->displayAverage();
        return ob_get_clean();
    }

    /**
     * Display a list of reviews in a gallery.
     *
     * @param array|string $atts List of attributes added to shortcode.
     *
     * @return string
     */
    public static function shortcodeGallery($atts = array()) {
        ob_start();
        (new static)->displayGallery($atts);
        return ob_get_clean();
    }

    /**
     * Fetch a list of reviews.
     *
     * @param array $args
     * @param bool  $pagination
     */
    public function initialize($args = array(), $pagination = true) {
        $args = wp_parse_args($args, array('post_type' => 'review', 'posts_per_page' => -1));
        $args = apply_filters('yndenz_modules_reviews_query_args', $args);

        $this->query = new WP_Query($args);

        if (!$this->query->have_posts()) {
            $this->initialized = true;
            return;
        }

        $result = new stdClass();
        $result->num_reviews = $this->query->found_posts;

        $reviews = $this->mapReviews();

        $result->average = array_sum(array_map(function ($item) {
            return $item->rating;
        }, $reviews)) / $result->num_reviews;

        $num_items = get_option('posts_per_page');
        if (array_key_exists('posts_per_page', $args)) {
            $num_items = $args['posts_per_page'];
        }
        $offset = 0;
        if ($pagination) {
            $page = get_query_var('paged', 1);
            $offset = ($page - 1) * $num_items;
        }
        $result->reviews = array_slice($reviews, $offset, $num_items);

        $result->item_reviewed = static::itemReviewed();

        $this->result = $result;
        $this->initialized = true;
    }

    protected function mapReviews() {
        $reviews = array();

        foreach ($this->query->posts as $review) {
            $item = new stdClass();
            $item->author = get_the_title($review);
            $item->date = get_the_date('Y-m-d', $review);
            $item->rating = (float)str_replace(',', '.', get_post_meta($review->ID, 'review_rating', true));
            $item->reason = get_post_meta($review->ID, 'review_role', true);
            $item->text = $review->post_content;
            array_push($reviews, $item);
        }

        return $reviews;
    }

    /**
     * Get the item reviewed.
     *
     * @return string
     */
    public static function itemReviewed() {
        $template = locate_template('modules/reviews-item-reviewed.php');

        if (empty($template)) {
            $template = locate_template('klantenvertellen-item-reviewed.php');
        }

        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/reviews-item-reviewed.php';
        }

        ob_start();
        include $template;
        return ob_get_clean();
    }

    /**
     * Display a list of reviews.
     *
     * @param array|string $args List of attributes added to shortcode.
     */
    public function displayList($args = array()) {
        if (!is_array($args)) {
            $args = shortcode_parse_atts($args);
        }

        if (!$this->initialized) {
            $this->initialize($args);
        }

        if (!isset($this->result)) {
            return;
        }

        $template = locate_template('modules/reviews-list.php');

        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/reviews-list.php';
        }

        include $template;
    }

    /**
     * Display the reviews average rating.
     */
    public function displayAverage() {
        if (!$this->initialized) {
            $this->initialize();
        }

        if (!isset($this->result)) {
            return;
        }

        $template = locate_template('modules/reviews-average.php');

        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/reviews-average.php';
        }

        include $template;
    }

    /**
     * Display a list of reviews in a gallery.
     *
     * @param array|string $args List of attributes added to shortcode.
     */
    public function displayGallery($args = array(), $items_per_slide = 1) {
        if (!is_array($args)) {
            $args = shortcode_parse_atts($args);
        }

        if (!$this->initialized) {
            if (!is_array($args)) {
                $args = array();
            }
            $args['posts_per_page'] = ceil(get_option('posts_per_page') / $items_per_slide);
            $this->initialize($args, false);
        }

        if (!isset($this->result)) {
            return;
        }

        $template = locate_template('modules/reviews-gallery.php');

        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/reviews-gallery.php';
        }

        include $template;
    }

    /**
     * Display a single review.
     *
     * @param stdClass $review
     * @param bool $full_content
     */
    public function display($review, $full_content = false, $show_read_more = true) {
        $template = locate_template('modules/reviews-review.php');

        if (empty($template)) {
            $template = dirname(__FILE__) . '/templates/reviews-review.php';
        }

        include $template;
    }

}
