<?php

namespace yndenz\Modules\yndenzSEO;

if ( is_dir( WP_PLUGIN_DIR . '/wordpress-seo' ) ) {
	require_once WP_PLUGIN_DIR . '/wordpress-seo/vendor/autoload.php';
} else if ( is_dir( WP_PLUGIN_DIR . '/wordpress-seo-premium' ) ) {
	require_once WP_PLUGIN_DIR . '/wordpress-seo-premium/vendor/autoload.php';
}
require_once dirname( __FILE__ ) . '/functions.php';

use WP_Post;
use WP_Term;
use WPSEO_Frontend;
use WPSEO_Options;
use WPSEO_Replace_Vars;
use yndenz\Modules\BaseModule\BaseModule;

class SEO extends BaseModule {

	public static function register() {
		add_filter( 'wpseo_breadcrumb_output', array( static::class, 'breadcrumbMicroformats' ) );
		add_filter( 'wpseo_json_ld_output', array( static::class, 'addAddressToJSONLD' ), 99, 2 );
	}

	/**
	 * Convert yoast breadcrumbs to Microformatted breadcrumbs
	 *
	 * @param $breadcrumbs
	 *
	 * @return mixed
	 */
	public static function breadcrumbMicroformats( $breadcrumbs ) {
		// Convert the breadcrumb to text without tags
		$breadcrumbs = strip_tags( $breadcrumbs, '<a>' );

		// Convert to array
		$separator = get_option( 'wpseo_internallinks' )['breadcrumbs-sep'];
		$items     = explode( $separator, $breadcrumbs );

		// Create a breadcrumb list
		array_walk( $items, array( static::class, 'breadcrumbConvertItem' ), $items );

		$itemprop = apply_filters( 'yndenz_modules_seo_breadcrumb_itemprop', 'itemprop="breadcrumb"' );
		$class    = apply_filters( 'yndenz_modules_seo_breadcrumb_class', 'breadcrumb' );

		if ( version_compare( WPSEO_VERSION, '7.7', '<' ) ) {
			$output = sprintf( '<ol %s itemscope itemtype="http://schema.org/BreadcrumbList" class="%s">%s</ol>', $itemprop, $class, implode( '', $items ) );
		} else {
			$output = sprintf( '<ul class="%s">%s</ul>', $class, implode( '', $items ) );
		}

		return apply_filters( 'yndenz_modules_seo_breadcrumb_output', $output );
	}

	/**
	 * Convert breadcrumb item into a list item.
	 *
	 * @param string $item
	 * @param int    $index
	 */
	public static function breadcrumbConvertItem( &$item, $index ) {
		$active = '';

		if ( strstr( $item, '<a' ) ) {
			static::breadcrumbConvertAnchor( $item );
		} else {
			$active = ' class="active"';
			static::breadcrumbConvertActiveItem( $item );
		}

		if ( version_compare( WPSEO_VERSION, '7.7', '<' ) ) {
			$item = sprintf( '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"%s>%s<meta itemprop="position" content="%d"/></li>', $active, trim( $item ), $index + 1 );
		} else {
			$item = sprintf( '<li%s>%s</li>', $active, trim( $item ) );
		}
	}

	/**
	 * Convert breadcrumb item anchor.
	 *
	 * @param string $item
	 */
	public static function breadcrumbConvertAnchor( &$item ) {
		$seo_title = '';

		if ( preg_match( '~title="([^"]+)"~i', $item, $matches ) > 0 ) {
			$seo_title = $matches[1];
		} else if ( preg_match( '~href="([^"]+)"~i', $item, $matches ) > 0 ) {
			$seo_title = static::title( null, $matches[1] );
		}

		if ( version_compare( WPSEO_VERSION, '7.7', '<' ) ) {
			$item = preg_replace( '/rel="v:url" property="v:title">([^<]+)</', 'itemprop="item" title="' . $seo_title . '"><span itemprop="name">$2</span><', $item );
		} else {
			$item = preg_replace( '/>([^<]+)</', ' title="' . $seo_title . '">$1<', $item );
		}
	}

	public static function addAddressToJSONLD( $data, $context ) {
		if ( $context === 'company' ) {
			$data['address'] = apply_filters( 'yndenz_modules_seo_json_ld_address', array(
				'@type'           => 'PostalAddress',
				'streetAddress'   => of_get_option( 'street' ) . ' ' . of_get_option( 'number' ),
				'postalCode'      => of_get_option( 'zip' ),
				'addressLocality' => of_get_option( 'city' ),
				'addressCountry'  => 'NL',
				'telephone'       => apply_filters( 'href_phone_number', of_get_option( 'phone' ) ),
				'email'           => get_option( 'admin_email' )
			) );
		}

		return $data;
	}

	/**
	 * Fetch the title for a specific object/URL
	 *
	 * @param WP_Post|WP_Term|null $object The object to get the SEO title for
	 * @param string|null          $url    The URL to fetch the SEO title for
	 *
	 * @return mixed|string The yoast SEO title or an empty string in case it could not be found
	 */
	public static function title( $object = null, $url = null ) {
		return static::get_meta( 'title', $object, $url );
	}

	/**
	 * Fetch the specific meta for a specific URL
	 *
	 * @param string               $type   The meta type to return, either 'title' or 'metadesc'
	 * @param WP_Post|WP_Term|null $object The object to fetch the meta for
	 * @param string|null          $url    The URL to fetch the meta for
	 *
	 * @return mixed|string The yoast SEO meta or an empty string in case it could not be found
	 */
	public static function get_meta( $type, $object = null, $url = null ) {
		$seo_meta = '%%title%% %%sep%% %%sitename%%';
		if ( method_exists( 'WPSEO_Options', 'get_options' ) ) {
			$seo_options = WPSEO_Options::get_options( array( 'wpseo', 'wpseo_titles' ) );
		} else {
			$seo_options = WPSEO_Options::get_all();
		}

		if ( is_null( $object ) ) {
			if ( is_null( $url ) ) {
				$object = get_queried_object();
			} else {
				$object = static::urlToObject( $url );
			}
		}

		if ( $object instanceof WP_Post ) {
			$seo_meta = static::postMeta( $type, $object, $seo_options );
		} else if ( $object instanceof WP_Term ) {
			$seo_meta = static::termMeta( $type, $object, $seo_options );
		} else {
			$post_types = get_post_types();
			foreach ( $post_types as $post_type ) {
				if ( $url === get_post_type_archive_link( $post_type ) ) {
					if ( array_key_exists( $type . '-ptarchive-' . $post_type, $seo_options ) ) {
						$seo_meta = $seo_options[ $type . '-ptarchive-' . $post_type ];
					}
				}
			}
		}

		if ( empty( $object ) ) {
			$object = array();
		}

		$seo_meta = ( new WPSEO_Replace_Vars )->replace( $seo_meta, $object );

		return apply_filters( 'wpseo_' . $type, $seo_meta );
	}

	/**
	 * Find the object found if you navigate to the given url.
	 *
	 * @param string $url
	 *
	 * @return WP_Post|WP_Term|null
	 */
	public static function urlToObject( $url ) {
		$object = null;
		$path   = parse_url( $url, PHP_URL_PATH );

		if ( trim( $url, '/' ) . '/' === home_url( '/' ) ) {
			$object = get_post( get_option( 'page_on_front' ) );
		} else {
			$object = get_page_by_path( $path, OBJECT, get_post_types() );
		}

		if ( ! $object instanceof WP_Post && $path !== '/' ) {
			$slug  = preg_replace( '~.*/([^/]+)/?$~i', '$1', $path );
			$terms = get_terms( array( 'taxonomy' => get_taxonomies(), 'slug' => $slug, 'hide_empty' => false ) );
			if ( is_array( $terms ) && count( $terms ) > 0 ) {
				$object = $terms[0];
			}
		}

		return $object;
	}

	/**
	 * Get the meta data for the given post object.
	 *
	 * @param string  $type
	 * @param WP_Post $post
	 * @param array   $seo_options
	 * @param string  $seo_meta
	 *
	 * @return string
	 */
	public static function postMeta( $type, $post, $seo_options, $seo_meta = '%%title%% %%sep%% %%sitename%%' ) {
		if ( $type === 'title' ) {
			$seo_frontend = WPSEO_Frontend::get_instance();
			$seo_meta     = $seo_frontend->get_content_title( $post );
		} else {
			if ( array_key_exists( $type . '-' . $post->post_type, $seo_options ) ) {
				$seo_meta = $seo_options[ $type . '-' . $post->post_type ];
			}
		}
		if ( empty( $seo_meta ) ) {
			if ( array_key_exists( $type . '-' . get_post_type( $post ), $seo_options ) ) {
				$seo_meta = $seo_options[ $type . '-' . get_post_type( $post ) ];
			}
		}

		return $seo_meta;
	}

	/**
	 * Get the meta data for the given term.
	 *
	 * @param        $type
	 * @param        $term
	 * @param        $seo_options
	 * @param string $seo_meta
	 *
	 * @return string
	 */
	public static function termMeta( $type, $term, $seo_options, $seo_meta = '%%term_title%% %%page%% %%sep%% %%sitename%%' ) {
		$short_type = str_replace( 'meta', '', $type );
		$meta       = get_option( 'wpseo_taxonomy_meta' );

		if ( $meta && array_key_exists( $term->taxonomy, $meta ) && array_key_exists( $term->term_id, $meta[ $term->taxonomy ] ) && array_key_exists( 'wpseo_' . $short_type, $meta[ $term->taxonomy ][ $term->term_id ] ) ) {
			$seo_meta = $meta[ $term->taxonomy ][ $term->term_id ][ 'wpseo_' . $short_type ];
		} else if ( array_key_exists( $type . '-tax-' . $term->taxonomy, $seo_options ) ) {
			$seo_meta = $seo_options[ $type . '-tax-' . $term->taxonomy ];
		}

		return $seo_meta;
	}

	/**
	 * Convert breadcrumb active item.
	 *
	 * @param string $item
	 */
	public static function breadcrumbConvertActiveItem( &$item ) {
		$seo_title = wp_title( '|', false );

		if ( is_search() ) {
			$permalink    = home_url( '/' ) . '?s=';
			$search_query = get_search_query();
			if ( ! empty( $search_query ) ) {
				$permalink .= get_search_query();
			} else {
				$item      = 'Zoeken';
				$seo_title = preg_replace( '~[^\|]+ \|~i', _x( 'Zoeken', 'Zoekresultaten', 'base-theme' ) . ' |', $seo_title );
			}
		} else {
			$permalink = get_permalink();
		}

		$item = sprintf( '<a href="%s" itemprop="item" title="%s"><span itemprop="name">%s</span></a>', $permalink, $seo_title, trim( $item ) );
	}

	/**
	 * Fetch the SEO description for a specific object/URL
	 *
	 * @param WP_Post|WP_Term|null $object The object to get the SEO description for
	 * @param string|null          $url    The URL to fetch the SEO description for
	 *
	 * @return mixed|string The yoast SEO description or an empty string in case it could not be found
	 */
	public static function description( $object = null, $url = null ) {
		return static::get_meta( 'metadesc', $object, $url );
	}

	/**
	 * Restructure content to include the correct microformats, HTML5-structure and styling classes.
	 *
	 * @param bool   $include_date
	 * @param string $after_title
	 *
	 * @return string
	 */
	public static function restructuredContent( $include_date = false, $after_title = '' ) {
		global $post;

		$content = $post->post_content;

		// Add the title if it's not already in the content
		if ( preg_match( '/<h1(.*)>(.*)<\/h1>/i', $post->post_content ) <= 0 ) :
			$content = '<h1>' . get_the_title() . '</h1>' . $content;
		endif;

		$content_filtered = apply_filters( 'the_content', $content );

		$title = '<h1 itemprop="name"$1>$2</h1>';

		$date = '';
		if ( $include_date ) {
			$datetime = get_the_time( 'Y-m-d' );
			$date     = sprintf( '<time datetime="%s" pubdate itemprop="datePublished" content="%s">%s</time>', $datetime, $datetime, get_the_date() );
			$date     .= sprintf( '<meta itemprop="dateModified" content="%s"/>', get_the_modified_time( 'Y-m-d' ) );
		}

		$page_header = sprintf( '<header class="page-header">%s</header>', $title . $date . $after_title );

		$content_restructured = preg_replace( '/<h1(.*)>(.*)<\/h1>/i', $page_header, $content_filtered );
		$content_restructured = preg_replace( '/<p([^>]*)>/i', '<p itemprop="description"$1>', $content_restructured, 1 );

		return apply_filters( 'yndenz_modules_seo_restructured_content', $content_restructured );
	}

}