<?php
function breadcrumbMicroformats ($breadcrumbs) {
    // Convert the breadcrumb to text without tags
    $breadcrumbs = strip_tags( $breadcrumbs, '<a>' );

    // Convert to array
    if (get_option('wpseo_titles')) {
        $separator = get_option('wpseo_titles')['breadcrumbs-sep'];
    } else {
        $separator = get_option('wpseo_internallinks')['breadcrumbs-sep'];
    }

    // Make an array of the breadcrumb items
    $items = explode($separator, $breadcrumbs);

    $formattedBreadcrumbs = '';

    // Create a breadcrumb list
    foreach ($items as $index => $item) {
        $active = '';

        // The active breadcrumb item doesn't have a link, so thats how we can set the active class
        if (strstr($item, '<a')) {
            $item = breadcrumbConvertAnchor($item);
        } else {
            $active = ' class="active"';
            $item = breadcrumbConvertActiveItem($item);
        }

        if ( version_compare( WPSEO_VERSION, '7.7', '<' ) ) {
            $item = sprintf( '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"%s>%s<meta itemprop="position" content="%d"/></li>', $active, trim( $item ), $index + 1 );
        } else {
            $item = sprintf( '<li%s>%s</li>', $active, trim( $item ) );
        }

        $formattedBreadcrumbs .= $item;
    }

    // Check which version of WPSEO is used to determine to use ol or ul
    if (version_compare(WPSEO_VERSION, '7.7', '<')) {
        $output = '<ol itemprop="breadcrumb" itemscope itemtype="http://schema.org/BreadcrumbList" class="breadcrumb">' . $formattedBreadcrumbs . '</ol>';
    } else {
        $output = '<ul class="breadcrumb">' . $formattedBreadcrumbs . '</ul>';
    }

    return $output;
}

add_filter('wpseo_breadcrumb_output', 'breadcrumbMicroformats');

function breadcrumbConvertAnchor( $item ) {
    $seo_title = '';

    if ( preg_match( '~title="([^"]+)"~i', $item, $matches ) > 0 ) {
        $seo_title = $matches[1];
    } else if ( preg_match( '~href="([^"]+)"~i', $item, $matches ) > 0 ) {
        $seo_title = title( null, $matches[1] );
    }

    if ( version_compare( WPSEO_VERSION, '7.7', '<' ) ) {
        $item = preg_replace( '/rel="v:url" property="v:title">([^<]+)</', 'itemprop="item" title="' . $seo_title . '"><span itemprop="name">$2</span><', $item );
    } else {
        $item = preg_replace( '/>([^<]+)</', ' title="' . $seo_title . '">$1<', $item );
    }

    return $item;
}

function title( $object = null, $url = null ) {
    return get_meta( 'title', $object, $url );
}

function get_meta( $type, $object = null, $url = null ) {
    $seo_meta = '%%title%% %%sep%% %%sitename%%';
    if ( method_exists( 'WPSEO_Options', 'get_options' ) ) {
        $seo_options = WPSEO_Options::get_options( array( 'wpseo', 'wpseo_titles' ) );
    } else {
        $seo_options = WPSEO_Options::get_all();
    }

    if ( is_null( $object ) ) {
        if ( is_null( $url ) ) {
            $object = get_queried_object();
        } else {
            $object = urlToObject( $url );
        }
    }

    if ( $object instanceof WP_Post ) {
        $seo_meta = postMeta( $type, $object, $seo_options );
    } else if ( $object instanceof WP_Term ) {
        $seo_meta = termMeta( $type, $object, $seo_options );
    } else {
        $post_types = get_post_types();
        foreach ( $post_types as $post_type ) {
            if ( $url === get_post_type_archive_link( $post_type ) ) {
                if ( array_key_exists( $type . '-ptarchive-' . $post_type, $seo_options ) ) {
                    $seo_meta = $seo_options[ $type . '-ptarchive-' . $post_type ];
                }
            }
        }
    }

    if ( empty( $object ) ) {
        $object = array();
    }

    $seo_meta = ( new WPSEO_Replace_Vars )->replace( $seo_meta, $object );

    return null;
}

function urlToObject( $url ) {
    $object = null;
    $path   = parse_url( $url, PHP_URL_PATH );

    if ( trim( $url, '/' ) . '/' === home_url( '/' ) ) {
        $object = get_post( get_option( 'page_on_front' ) );
    } else {
        $object = get_page_by_path( $path, object, get_post_types() );
    }

    if ( ! $object instanceof WP_Post && $path !== '/' ) {
        $slug  = preg_replace( '~.*/([^/]+)/?$~i', '$1', $path );
        $terms = get_terms( array( 'taxonomy' => get_taxonomies(), 'slug' => $slug, 'hide_empty' => false ) );
        if ( is_array( $terms ) && count( $terms ) > 0 ) {
            $object = $terms[0];
        }
    }

    return $object;
}

function postMeta( $type, $post, $seo_options, $seo_meta = '%%title%% %%sep%% %%sitename%%' ) {
    if ( $type === 'title' ) {
        if ( version_compare( WPSEO_VERSION, '17.8', '<' ) ) {
            $seo_frontend = WPSEO_Frontend::get_instance();
            if ( version_compare( WPSEO_VERSION, '14.0', '<' ) ) {
                $seo_meta = $seo_frontend->get_content_title( $post );
            }
        }
    } else {
        if ( array_key_exists( $type . '-' . $post->post_type, $seo_options ) ) {
            $seo_meta = $seo_options[ $type . '-' . $post->post_type ];
        }
    }
    if ( empty( $seo_meta ) ) {
        if ( array_key_exists( $type . '-' . get_post_type( $post ), $seo_options ) ) {
            $seo_meta = $seo_options[ $type . '-' . get_post_type( $post ) ];
        }
    }

    return $seo_meta;
}

function termMeta( $type, $term, $seo_options, $seo_meta = '%%term_title%% %%page%% %%sep%% %%sitename%%' ) {
    $short_type = str_replace( 'meta', '', $type );
    $meta       = get_option( 'wpseo_taxonomy_meta' );

    if ( $meta && array_key_exists( $term->taxonomy, $meta ) && array_key_exists( $term->term_id, $meta[ $term->taxonomy ] ) && array_key_exists( 'wpseo_' . $short_type, $meta[ $term->taxonomy ][ $term->term_id ] ) ) {
        $seo_meta = $meta[ $term->taxonomy ][ $term->term_id ][ 'wpseo_' . $short_type ];
    } else if ( array_key_exists( $type . '-tax-' . $term->taxonomy, $seo_options ) ) {
        $seo_meta = $seo_options[ $type . '-tax-' . $term->taxonomy ];
    }

    return $seo_meta;
}

function breadcrumbConvertActiveItem( $item ) {
    $seo_title = wp_title( '|', false );

    if ( is_search() ) {
        $permalink    = home_url( '/' ) . '?s=';
        $search_query = get_search_query();
        if ( ! empty( $search_query ) ) {
            $permalink .= get_search_query();
        } else {
            $item      = 'Zoeken';
            $seo_title = preg_replace( '~[^\|]+ \|~i', _x( 'Zoeken', 'Zoekresultaten', 'base-theme' ) . ' |', $seo_title );
        }
    } else {
        $queried_object = get_queried_object();

        if ( $queried_object instanceof WP_Post ) {
            $permalink = get_permalink();
        } else if ( $queried_object instanceof WP_Term ) {
            $permalink = get_term_link( $queried_object, $queried_object->taxonomy );
        } else if ( is_post_type_archive() && ! empty( get_query_var( 'post_type' ) ) ) {
            $permalink = get_post_type_archive_link( get_query_var( 'post_type' ) );
        } else {
            global $wp;
            $permalink = home_url( $wp->request );
        }
    }

    if ( version_compare( WPSEO_VERSION, '7.7', '<' ) ) {
        $item = sprintf( '<a href="%s" itemprop="item" title="%s"><span itemprop="name">%s</span></a>', $permalink, $seo_title, trim( $item ) );
    } else {
        $item = sprintf( '<a href="%s" title="%s">%s</a>', $permalink, $seo_title, trim( $item ) );
    }

    return $item;
}

function restructuredContent( $include_date = false, $after_title = '' ) {
    global $post;

    $content = $post->post_content;

    // Add the title if it's not already in the content
    if ( preg_match( '/<h1(.*)>(.*)<\/h1>/i', $post->post_content ) <= 0 ) :
        $content = '<h1>' . get_the_title() . '</h1>' . $content;
    endif;

    $content_filtered = apply_filters( 'the_content', $content );

    if ( version_compare( WPSEO_VERSION, '11.0', '<' ) ) {
        $title = '<h1 itemprop="name"$1>$2</h1>';
    } else {
        $title = '<h1$1>$2</h1>';
    }

    $date = '';
    if ( $include_date ) {
        $datetime = get_the_time( 'Y-m-d' );
        if ( version_compare( WPSEO_VERSION, '11.0', '<' ) ) {
            $date = sprintf( '<time datetime="%s" pubdate itemprop="datePublished" content="%s">%s</time>', $datetime, $datetime, get_the_date() );
            $date .= sprintf( '<meta itemprop="dateModified" content="%s"/>', get_the_modified_time( 'Y-m-d' ) );
        } else {
            $date = sprintf( '<time datetime="%s">%s</time>', $datetime, get_the_date() );
        }
    }

    $page_header = sprintf( '<header class="page-header">%s</header>', $title . $date . $after_title );

    $content_restructured = preg_replace( '/<h1(.*)>(.*)<\/h1>/i', $page_header, $content_filtered );
    if ( version_compare( WPSEO_VERSION, '11.0', '<' ) ) {
        $content_restructured = preg_replace( '/<p([^>]*)>/i', '<p itemprop="description"$1>', $content_restructured, 1 );
    }

    return $content_restructured;
}