<?php

namespace BaseTheme\Basics;

use DOMDocument;
use DOMXPath;
use PHPMailer;
use WP_Post;

class Reset {

    public static function init() {
        // Remove WordPress version number in <head> section
        remove_action('wp_head', 'wp_generator');

        // Disable XML-RPC as we don't want to access and publish remotely
        add_filter('xmlrpc_enabled', '__return_false');

        // Disable auto-updating of plugins
        add_filter('auto_update_plugin', '__return_false');

        // Disable automatic embedding by WordPress
        add_action('init', array(static::class, 'disableEmbed'));

        // Disable Emoji's
        add_action('init', array(static::class, 'disableEmoji'));

        // Fix errors on multipart email bodies
        add_action('phpmailer_init', array(static::class, 'fixMultipartEmail'));

        // Add the title to the attachments
        add_filter('wp_get_attachment_image_attributes', array(static::class, 'addTitleToAttachments'), 99, 2);

        // Hide source output comment for "W3 Total Cache" plugin
        add_filter('w3tc_can_print_comment', '__return_false', 10, 1);
    }

    /**
     * Disable automatic embedding by WordPress.
     */
    public static function disableEmbed() {
        remove_action('rest_api_init', 'wp_oembed_register_route');
        remove_filter('oembed_dataparse', 'wp_filter_oembed_result', 10);
        remove_action('wp_head', 'wp_oembed_add_discovery_links');
        remove_action('wp_head', 'wp_oembed_add_host_js');
    }

    /**
     * Disable emoji's
     */
    public static function disableEmoji() {
        // all actions related to emojis
        remove_action('admin_print_styles', 'print_emoji_styles');
        remove_action('wp_head', 'print_emoji_detection_script', 7);
        remove_action('admin_print_scripts', 'print_emoji_detection_script');
        remove_action('wp_print_styles', 'print_emoji_styles');
        remove_filter('wp_mail', 'wp_staticize_emoji_for_email');
        remove_filter('the_content_feed', 'wp_staticize_emoji');
        remove_filter('comment_text_rss', 'wp_staticize_emoji');
        add_filter( 'tiny_mce_plugins', array(static::class, 'disableEmojisTinyMCE') );
        add_filter('wp_resource_hints', array(static::class, 'removeDnsPrefetch'), 10, 2);
    }

    /**
     * Filter function used to remove the tinymce emoji plugin.
     *
     * @param array $plugins
     * @return array Difference betwen the two arrays
     */
    public static function disableEmojisTinyMCE( $plugins ) {
        if ( is_array( $plugins ) ) {
            return array_diff( $plugins, array( 'wpemoji' ) );
        } else {
            return array();
        }
    }

    /**
     * Remove the s.w.org dns prefetch that's only used for emoji's (which we don't allow anyway)
     *
     * @param array $hints
     * @param string $relation_type
     *
     * @return array
     */
    public static function removeDnsPrefetch($hints, $relation_type) {
        if ('dns-prefetch' === $relation_type) {
            $matches = preg_grep('/emoji/', $hints);
            return array_diff($hints, $matches);
        }

        return $hints;
    }

    /**
     * Fixes the fact that Wordpress doesn't automatically add text/plain version to html e-mail
     *
     * @param PHPMailer $phpmailer
     */
    public static function fixMultipartEmail($phpmailer) {
        $doc = new DOMDocument();
        @$doc->loadHTML($phpmailer->Body);

        $xpath = new DOMXPath($doc);

        foreach ($xpath->query('//style') as $node) {
            $node->parentNode->removeChild($node);
        }

        $html = $doc->saveHTML();

        $phpmailer->AltBody = strip_tags($html);
    }

    /**
     * Add title attribute to images if it's not already there.
     * Defaults to use alt as title.
     *
     * @param array $atts
     * @param WP_Post $attachment
     *
     * @return array
     */
    public static function addTitleToAttachments($atts, $attachment) {
        if (!array_key_exists('title', $atts)) {
            $atts['title'] = get_the_title($attachment);
        }
        if (empty($atts['alt'])) {
            $atts['alt'] = $atts['title'];
        }

        return $atts;
    }

}