<?php

namespace BaseTheme\Basics;

class ThemeSupport extends Basics {

	public static function register() {
		add_theme_support( 'menus' );
		add_theme_support( 'post-thumbnails' );
		add_filter( 'widget_text', 'do_shortcode' );

		add_filter( 'logout_url', array( static::class, 'forceLogoutRedirectToHome' ), 102, 2 );
	}

	/**
	 * Force redirect to base url after logout, because it'll redirect you to login by default resulting in a 404.
	 *
	 * @param string $logout_url
	 *
	 * @return string
	 */
	public static function forceLogoutRedirectToHome( $logout_url ) {
		return preg_replace( '/&amp;redirect_to([^&]+)/', '&amp;redirect_to=' . urlencode( home_url() ), $logout_url );
	}

	/**
	 * Get the company logo image tag.
	 *
	 * @param string $size
	 * @param null   $title
	 *
	 * @return string|bool HTML-img, or false if no company logo was set.
	 */
	public static function getCompanyLogo( $size = 'medium', $title = null ) {
		$company_logo_id = static::getCompanyLogoID();
		if ( $company_logo_id === false ) {
			return false;
		}

		if ( empty( $title ) ) {
			$front_page = get_post( get_option( 'page_on_front' ) );
			get_seo_title( $front_page );
		}

		return wp_get_attachment_image( $company_logo_id, $size, false, array( 'alt' => $title, 'title' => $title ) );
	}

	/**
	 * Get company logo ID.
	 *
	 * @return int|bool ID of the media file, or false if no company logo was set.
	 */
	public static function getCompanyLogoID() {
		$company_logo_id = of_get_option( 'company_logo' );

		if ( empty( $company_logo_id ) ) {
			return false;
		}

		if ( ! is_numeric( $company_logo_id ) ) {
			$company_logo_id = get_attachment_id_from_url( $company_logo_id );
		}

		return intval( $company_logo_id );
	}

	/**
	 * Get the URL for the company logo.
	 *
	 * @param string $size
	 *
	 * @return string|bool URL of the company logo, or false if no company logo was set.
	 */
	public static function getCompanyLogoURL( $size = 'medium' ) {
		$company_logo_id = static::getCompanyLogoID();
		if ( $company_logo_id === false ) {
			return false;
		}

		$image = wp_get_attachment_image_src( $company_logo_id, $size, false );

		return $image ? $image[0] : false;
	}

	/**
	 * Get the archive title.
	 *
	 * @return null|string
	 */
	public static function getArchiveTitle() {
		if ( is_date() ) {
			if ( is_category() ) {
				return single_cat_title( '', false ) . single_month_title( ' ', false );
			} else {
				return trim( single_month_title( ' ', false ) );
			}
		} elseif ( is_category() || is_tag() ) {
			return single_tag_title( '', false );
		}

		return null;
	}

}