<?php
/*
Plugin Name: Feedback Company
Plugin URI:  http://www.yndenz.com
Description: Feedbackcompany.com page reader for reviews
Version:     1.0.4
Author:      Gido Manders <g.manders@yndenz.com>
Author URI:  http://www.yndenz.com
Domain Path: /languages
Text Domain: feedback-company
*/

/**
 * Add a menu item to the settings menu
 */
add_action('admin_menu', 'feedback_company_menu');
function feedback_company_menu()
{
    // Add a new submenu under Settings
    add_options_page(__('Feedback Company', 'feedback-company'), __('Feedback Company', 'feedback-company'), 'manage_options', 'feedback-company', 'feedback_company_settings_page');
}

/**
 * Display the settings page
 */
function feedback_company_settings_page()
{
    //must check that the user has the required capability
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.'));
    }

    // See if the user has saved the settings
    if (isset($_POST['feedback_company_slug'])) {
        // Save the posted value in the database
        update_option('feedback_company_slug', $_POST['feedback_company_slug']);
        ?>
        <div class="updated"><p><strong><?php _e('Settings saved.', 'feedback-company'); ?></strong></p></div>
        <?php
    }
    ?>
    <div class="wrap">
        <h2><?php _e('Feedback Company Settings', 'feedback-company'); ?></h2>
        <form name="feedbackCompanySettingsForm" method="post" action="">
            <label>
                <?php _e("Slug:", 'feedback-company'); ?>
            </label>
            <input type="text" name="feedback_company_slug" value="<?php echo get_option('feedback_company_slug'); ?>"/>
            <button type="submit" class="button-primary"><?php _e('Save', 'feedback-company'); ?></button>
        </form>
    </div>
    <?php
}

/**
 * Get all feedback-company reviews.
 * Define FEEDBACK_COMPANY_CACHE if you want to cache the reviews (updated hourly)
 *
 * @return object|boolean Returns an array containing the reviews, or false if an error occurred.
 */
function feedback_company_get_reviews($cron = false, $item_reviewed_template = '')
{
    // If the cache file is not defined or the file is updated over an hour ago, fetch the reviews
    if ($cron && !defined('FEEDBACK_COMPANY_CACHE') || $cron && !file_exists(FEEDBACK_COMPANY_CACHE) || $cron && filemtime(FEEDBACK_COMPANY_CACHE) < strtotime('-1 hour')) {

        $base_url = 'https://www.feedbackcompany.com/nl-nl/reviews/' . get_option('feedback_company_slug');
        try {
            // Ignore invalid HTML warnings
            libxml_use_internal_errors(true);

            $dom = new DomDocument();
            @$dom->loadHTMLFile($base_url);
            $xpath = new DOMXPath($dom);

            $result = new stdClass();
            // Set the average
            $average = $xpath->query('//header//*[contains(@class,"rating")]//*[contains(@class,"rating__numbers")]/span[1]/text()');
            $average_as_string = trim((string)$average->item(0)->nodeValue);
            $result->average = (float)str_replace(',', '.', $average_as_string);
            $num_reviews = $xpath->query('//header//*[contains(@class,"rating")]//*[contains(@class,"rating__review-count")]/text()')->item(0)->nodeValue;
            $num_reviews = intval(preg_replace('/\D/', '', $num_reviews));
            $result->num_reviews = number_format($num_reviews,0,',','.');
            $result->percentage_recommending = substr(trim($xpath->query('//*[contains(@class,"detailed-rating")]//*[contains(@class,"percentage-bar")]/@aria-valuenow')->item(0)->nodeValue), 0, 2);

            // Check whether the number of reviews in cache file and remote location match
            // It is highly probable that an update is not needed if these values match
            // So if they do we return the cached result immediately
            if(file_exists(FEEDBACK_COMPANY_CACHE)){
                $cached_result = json_decode(file_get_contents(FEEDBACK_COMPANY_CACHE));
                if($cached_result->num_reviews == $result->num_reviews)
                    return $cached_result;
            }

            // Loop the pages and convert the reviews
            $result->reviews = [];
            $i = 0;
            while ($i < ceil($num_reviews / 10)) {
                if ($i == 0) {
                    $page = $dom;
                } else {
                    $page = new DomDocument();
                    @$page->loadHTMLFile($base_url . '?starter=' . ($i * 10));
                }
                $page_xpath = new DOMXPath($page);

                // Loop the reviews to convert them into a readable item
                $reviews = $page_xpath->query('//*[contains(@class,"reviews")]//*[@class="review"]');
                if ($reviews->length === 0) {
                    break;
                }

                foreach ($reviews as $review) {
	                $rating           = $page_xpath->query( './/*[contains(@class ,"review__stars-number")]/text()', $review );
                    $rating_as_string = trim((string)$rating->item(0)->nodeValue);
                    if ($rating->length === 0 || $rating_as_string === 'n.v.t.') {
                        continue;
                    }
                    $rating_as_string = str_replace('/10', '', $rating_as_string);

                    $item            = new stdClass();
	                $item->author    = trim( (string) $page_xpath->query( './/*[contains(@class,"review__header__name")]/text()', $review )->item( 0 )->nodeValue );
	                $date            = trim( (string) $page_xpath->query( './/*[contains(@class,"review__timeline")]/time/text()', $review )->item( 0 )->nodeValue );
                    $item->date      = implode('-',array_reverse(explode('-',$date)));
                    $item->rating    = (float)str_replace(',', '.', $rating_as_string);
                    $item->recommend = strpos((string)$page_xpath->query('.//*[contains(@class,"review__recommendation-answer")]/text()', $review)->item(0)->nodeValue, 'Ja') >= 0;
                    $item->reason    = trim((string)$page_xpath->query('.//*[contains(@class,"review__strong-points")]/*[contains(@class,"paragraph")]/text()', $review)->item(0)->nodeValue);
                    $description     = $page_xpath->query('.//*[contains(@class,"review__content")]/*[contains(@class,"header")]/text()', $review);
                    $item->text      = '';
                    if ($description->length > 0) {
                        foreach ($description as $child) {
	                        $item->text .= trim( $child->nodeValue );
                        }
                    }

                    if (isset($cached_result)) {
                        // if the loop encounters a review that already exists
                        // every review after that is probably already in the cached
                        // This is an exact object match
                        foreach ($cached_result->reviews as $creview):
                            if($item == $creview)
                                break 3;
                        endforeach;
                    }

                    $result->reviews[] = $item;
                }

                $i++;
            }

            if (isset($cached_result)) {
                $result->reviews = array_merge($result->reviews,$cached_result->reviews);
            }
            
            $result->found_reviews = count($result->reviews);

            // If the cache file is defined, save the results JSON encoded in the file
            if ($result->found_reviews > 0) {
                if (defined('FEEDBACK_COMPANY_CACHE')) {
                    file_put_contents(FEEDBACK_COMPANY_CACHE, json_encode($result));
                }
            } else {
                mail('developers@yndenz.com',get_bloginfo('title').' 0 Reviews',$_SERVER['HTTP_HOST']);
                throw new Exception(__('No reviews found', 'feedback-company'));
            }

            ob_start();
	        include locate_template('feedback-company-item-reviewed.php');
	        $result->item_reviewed = ob_get_clean();

            return $result;
        } catch (Exception $e) {
            if (WP_DEBUG) {
                var_dump($e);
            }

            // If the cache file is defined and the file exists, return the cached results to prevent empty results
            // This could help in case feedbackcompany.com is down or an error occurred while reading a page
            if (defined('FEEDBACK-COMPANY_CACHE') && file_exists(FEEDBACK_COMPANY_CACHE)) {
                $result = json_decode(file_get_contents(FEEDBACK_COMPANY_CACHE));

	            ob_start();
	            if (empty($item_reviewed_template)) {
	                $item_reviewed_template = locate_template('feedback-company-item-reviewed.php');
                }
	            include $item_reviewed_template;
	            $result->item_reviewed = ob_get_clean();

	            return $result;
            }

            // An error occurred, so we return false
            return false;
        }
    } else {
        // Read the cache file and decode the contents to return the results as an object
        $result = json_decode(file_get_contents(FEEDBACK_COMPANY_CACHE));

	    ob_start();
        if (empty($item_reviewed_template)) {
            $item_reviewed_template = locate_template('feedback-company-item-reviewed.php');
        }
        include $item_reviewed_template;
	    $result->item_reviewed = ob_get_clean();

	    return $result;
    }
}

/**
 * Get average feedbackcompany.com rating
 */
add_shortcode('feedback_company_average', 'feedback_company_average');
function feedback_company_average()
{
    $klantvertellen = feedback_company_get_reviews();
    return $klantvertellen->average;
}

/**
 * Get total feedbackcompany.com reviews
 */
add_shortcode('feedback_company_total', 'feedback_company_total');
function feedback_company_total()
{
    $klantvertellen = feedback_company_get_reviews();
    return $klantvertellen->found_reviews;
}
