class CookieConsent {
    // Update status constants
    constructor(config) {

        // but ensure all required properties exist
        this.config = {
            enabled: true,
            container: null,
            cookie: {
                name: 'PrivacyCookieNotice',
                path: '/',
                domain: '',
                expiryDays: 365,
                secure: false,
                ...(config?.cookie || {})
            },
            content: {
                header: 'Cookies used on the website!',
                message: 'This website uses cookies to ensure you get the best experience on our website.',
                dismiss: 'Got it!',
                allow: 'Allow cookies',
                deny: 'Decline',
                link: 'Learn more',
                href: 'https://www.cookiesandyou.com',
                close: '&#x274c;',
                target: '_blank',
                policy: 'Cookie Policy',
                formValidationMessage: 'You must accept cookies to submit this form',
                ...(config?.content || {})
            },
            position: config?.position || 'bottom',
            theme: config?.theme || 'block',
            layout: config?.layout || 'basic',
            compliance: {
                type: config?.type || 'info',
            }
        };
    
        this.status = {
            ALLOW: 'dismiss',
            DENY: 'deny'
        };

        this.initialize();
    }

    // Update form submission check
    handleFormSubmit(e) {
        // Skip our handling for Contact Form 7 forms that have validation errors
        if (e.target.classList.contains('wpcf7-form')) {
            const invalidFields = e.target.querySelectorAll('.wpcf7-not-valid');
            if (invalidFields.length > 0) {
                return; // Let CF7 handle its own validation
            }
        }
        
        const status = this.getStatus();
        if (status !== 'dismiss') {
            e.preventDefault();
            this.pendingForm = e.target;
            this.showModal(true);
        }
    }

    submitPendingForm() {
        if (this.pendingForm) {
            // For Contact Form 7 forms, use a different approach
            if (this.pendingForm.classList.contains('wpcf7-form')) {
                // Find the submit button
                const submitButton = this.pendingForm.querySelector('input[type="submit"]');
                
                // Create a clone of the button to avoid event listeners
                if (submitButton) {
                    const cloneButton = submitButton.cloneNode(true);
                    submitButton.parentNode.replaceChild(cloneButton, submitButton);
                    
                    // Temporarily remove our event listener
                    const oldHandler = this.handleFormSubmit;
                    document.removeEventListener('submit', oldHandler);
                    
                    // Submit the form directly
                    setTimeout(() => {
                        this.pendingForm.submit();
                        
                        // Restore our event listener after form is submitted
                        setTimeout(() => {
                            document.addEventListener('submit', (e) => this.handleFormSubmit(e));
                        }, 500);
                    }, 100);
                }
            } else {
                // For non-CF7 forms
                this.pendingForm.submit();
            }
            this.pendingForm = null;
        }
    }

    // Update consent check
    hasConsented() {
        const status = this.getStatus();
        return status === this.status.ALLOW;
    }

    // Remove the deepMerge method as it's no longer needed
    deepMerge(target, source) {
        if (!source) return target;
        
        const result = { ...target };
        
        Object.keys(source).forEach(key => {
            if (source[key] && typeof source[key] === 'object' && !Array.isArray(source[key])) {
                if (!(key in target)) {
                    result[key] = source[key];
                } else {
                    result[key] = this.deepMerge(target[key], source[key]);
                }
            } else if (source[key] !== undefined) {
                result[key] = source[key];
            }
        });
        
        return result;
    }

    initialize() {
        if (!this.config.enabled) return;
        
        this.createStyles();
        this.createModal();
        this.attachEventListeners();
        
        // Initially hide the modal
        this.hideModal();
        
        // Check cookie status and show if needed
        const status = this.getStatus();
        if (!this.hasConsented()) {  // Changed to use hasConsented() check
            this.showModal();
        }
    }

    showModal(centered = false) {
        if (centered) {
            this.modal.classList.add('cc-center');
            this.overlay.style.display = 'block';
        }
        this.modal.classList.remove('cc-invisible');
    }

    hideModal() {
        this.modal.classList.add('cc-invisible');
        this.modal.classList.remove('cc-center');
        this.overlay.style.display = 'none';
    }

    setStatus(status) {
        const { name, path, domain, expiryDays, secure } = this.config.cookie;
        const expires = new Date();
        expires.setDate(expires.getDate() + expiryDays);

        document.cookie = `${name}=${status};expires=${expires.toUTCString()};path=${path}${domain ? `;domain=${domain}` : ''}${secure ? ';secure' : ''}`;
    }

    getStatus() {
        const match = document.cookie.match(new RegExp(`${this.config.cookie.name}=([^;]+)`));
        return match ? match[1] : null;
    }

    hasConsented() {
        const status = this.getStatus();
        return status === this.status.ALLOW;
    }

    submitPendingForm() {
        if (this.pendingForm) {
            // For Contact Form 7 forms
            if (this.pendingForm.classList.contains('wpcf7-form')) {
                const formWrapper = this.pendingForm.closest('.wpcf7');
                if (formWrapper) {
                    // Remove our submit event listener temporarily
                    document.removeEventListener('submit', this.handleFormSubmit);
                    
                    // Let CF7 handle the submission
                    const submitEvent = new Event('submit', {
                        bubbles: true,
                        cancelable: true
                    });
                    this.pendingForm.dispatchEvent(submitEvent);
                    
                    // Reattach our submit listener after a short delay
                    setTimeout(() => {
                        document.addEventListener('submit', (e) => this.handleFormSubmit(e));
                    }, 100);
                }
            } else {
                // For non-CF7 forms
                this.pendingForm.submit();
            }
            this.pendingForm = null;
        }
    }

    createStyles() {
        const styles = `
            .cc-window {
                position: fixed;
                overflow: hidden;
                box-sizing: border-box;
                font-size: 16px;
                line-height: 1.5em;
                display: flex;
                flex-wrap: nowrap;
                z-index: 9999;
                padding: 1em;
                background-color: rgba(20, 20, 20, 0.8);
                color: #fff;
            }
            .cc-window.cc-invisible {
                display: none;
            }
            .cc-window.cc-center {
                position: fixed;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%);
                max-width: 500px;
                border-radius: 4px;
            }
            .cc-bottom {
                bottom: 0;
                left: 0;
                right: 0;
            }
            .cc-top {
                top: 0;
                left: 0;
                right: 0;
            }
            .cc-bottom-left {
                bottom: 1em;
                left: 1em;
                max-width: 24em;
            }
            .cc-bottom-right {
                bottom: 1em;
                right: 1em;
                max-width: 24em;
            }
            .cc-message {
                margin-right: 1em;
                flex: 1;
            }
            .cc-compliance {
                display: flex;
                align-items: center;
                gap: 0.5em;
            }
            .cc-btn {
                padding: 0.5em 1em;
                border: none;
                border-radius: 4px;
                cursor: pointer;
                font-size: 1em;
                transition: background-color 0.2s;
            }
            .cc-allow {
                background-color: #28a745;
                color: #fff;
            }
            .cc-allow:hover {
                background-color: #218838;
            }
            .cc-deny {
                background-color: #dc3545;
                color: #fff;
            }
            .cc-deny:hover {
                background-color: #c82333;
            }
            .cc-link {
                color: #fff;
                text-decoration: underline;
                margin-left: 1em;
            }
            .cc-overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 9998;
            }
        `;

        const styleElement = document.createElement('style');
        styleElement.textContent = styles;
        document.head.appendChild(styleElement);
    }

    getComplianceContent() {
        switch (this.config.compliance?.type) {
            case 'opt-out':
                return `
                    <button class="cc-btn cc-allow">${this.config.content.allow}</button>
                    ${this.config.content.deny ? `<button class="cc-btn cc-deny">${this.config.content.deny}</button>` : ''}
                `;
            case 'opt-in':
                return `
                    ${this.config.content.deny ? `<button class="cc-btn cc-deny">${this.config.content.deny}</button>` : ''}
                    <button class="cc-btn cc-allow">${this.config.content.allow}</button>
                `;
            case 'info':
                return `<button class="cc-btn cc-allow">${this.config.content.dismiss}</button>`;
            default:
                return `<button class="cc-btn cc-allow">${this.config.content.dismiss}</button>`;
        }
    }

    createModal() {
        const complianceContent = this.getComplianceContent();
        const modalHtml = `
            <div class="cc-window cc-invisible ${this.getPositionClass()}" role="dialog" aria-label="cookieconsent">
                <div class="cc-message">${this.config.content.message}</div>
                <div class="cc-compliance">
                    ${complianceContent}
                </div>
                ${this.config.content.link ? `
                    <a href="${this.config.content.href}" target="${this.config.content.target}" class="cc-link">${this.config.content.link}</a>
                ` : ''}
            </div>
            <div class="cc-overlay"></div>
        `;

        const modalElement = document.createElement('div');
        modalElement.innerHTML = modalHtml;
        document.body.appendChild(modalElement);

        this.modal = modalElement.querySelector('.cc-window');
        this.overlay = modalElement.querySelector('.cc-overlay');
    }

    attachEventListeners() {
        document.addEventListener('submit', (e) => this.handleFormSubmit(e));

        this.modal.querySelector('.cc-allow').addEventListener('click', () => {
            this.setStatus(this.status.ALLOW);
            this.hideModal();
            this.submitPendingForm();
        });

        if (this.modal.querySelector('.cc-deny')) {
            this.modal.querySelector('.cc-deny').addEventListener('click', () => {
                this.setStatus(this.status.DENY);
                this.hideModal();
            });
        }
    }

    getPositionClass() {
        const positions = {
            'bottom': 'cc-bottom',
            'top': 'cc-top',
            'bottom-left': 'cc-bottom-left',
            'bottom-right': 'cc-bottom-right'
        };
        return `cc-theme-${this.config.theme} ${positions[this.config.position] || 'cc-bottom'}`;
    }
}

// Initialize when the DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    if (window.cookieconsent_config) {
        window.cookieconsent = new CookieConsent(window.cookieconsent_config);
    }
});