class CookieConsent {
    // Update status constants
    constructor(config) {
        // Parse custom attributes if they exist
        let customAttributes = {};
        if (config?.customAttributes) {
            try {
                customAttributes = typeof config.customAttributes === 'string'
                    ? JSON.parse(config.customAttributes)
                    : config.customAttributes;
            } catch (e) {
                // Keep empty catch for error handling
            }
        }

        // but ensure all required properties exist
        this.config = {
            enabled: true,
            container: null,
            no_consent_required_for_form_submission: config?.no_consent_required_for_form_submission || customAttributes?.no_consent_required_for_form_submission || false,
            cookie: {
                name: 'PrivacyCookieNotice',
                path: '/',
                domain: '',
                expiryDays: 365,
                secure: false,
                ...(customAttributes?.cookie || {}),
                ...(config?.cookie || {})
            },
            palette: {
                popup: {
                    background: 'rgba(20, 20, 20, 0.8)',
                    text: '#fff'
                },
                button: {
                    background: '#28a745',
                    text: '#fff',
                    border: 'transparent'
                },
                highlight: {
                    background: '#f1d600',
                    text: '#000000',
                    border: 'transparent'
                },
                ...(customAttributes?.palette || {}),
                ...(config?.palette || {})
            },
            content: {
                header: 'Cookies used on the website!',
                message: config?.content.message || 'This website uses cookies to ensure you get the best experience on our website.',
                dismiss: config?.content.dismiss || 'Got it!',
                allow: config?.allow || 'Allow cookies',
                deny: config?.deny || 'Decline',
                link: config?.link || 'Learn more',
                href: config?.policyhref || 'https://www.cookiesandyou.com',
                close: '&#x274c;',
                target: '_blank',
                policy: 'Cookie Policy',
                formValidationMessage: 'You must accept cookies to submit this form',
                ...(config?.content || {})
            },
            position: 'bottom-left',
            theme: 'block',
            layout: 'basic',
            compliance: {
                type: config?.type || 'info',
            }
        };

        this.status = {
            ALLOW: 'dismiss',
            DENY: 'deny'
        };

        this.initialize();
    }

    // Update form submission check
    handleFormSubmit(e) {
        if (this.config.no_consent_required_for_form_submission === true) {
            return;
        }

        const status = this.getStatus();
        if (status !== 'dismiss') {
            e.preventDefault();
            this.pendingForm = e.target;
            this.showModal(true);
        }
    }

    submitPendingForm() {
        if (this.pendingForm) {
            if (this.pendingForm.classList.contains('wpcf7-form')) {
                const submitButton = this.pendingForm.querySelector('input[type="submit"]');

                if (submitButton) {
                    const cloneButton = submitButton.cloneNode(true);
                    submitButton.parentNode.replaceChild(cloneButton, submitButton);

                    const oldHandler = this.handleFormSubmit;
                    document.removeEventListener('submit', oldHandler);

                    setTimeout(() => {
                        this.pendingForm.submit();

                        setTimeout(() => {
                            document.addEventListener('submit', (e) => this.handleFormSubmit(e));
                        }, 500);
                    }, 100);
                }
            } else {
                this.pendingForm.submit();
            }
            this.pendingForm = null;
        }
    }

    setStatus(status) {
        const { name, path, domain, expiryDays, secure } = this.config.cookie;
        const expires = new Date();
        expires.setDate(expires.getDate() + expiryDays);

        document.cookie = `${name}=${status};expires=${expires.toUTCString()};path=${path}${domain ? `;domain=${domain}` : ''}${secure ? ';secure' : ''}`;
    }

    getStatus() {
        const match = document.cookie.match(new RegExp(`${this.config.cookie.name}=([^;]+)`));
        return match ? match[1] : null;
    }

    hasConsented() {
        const status = this.getStatus();
        return status === this.status.ALLOW;
    }

    // Remove the deepMerge method as it's no longer needed
    deepMerge(target, source) {
        if (!source) return target;

        const result = { ...target };

        Object.keys(source).forEach(key => {
            if (source[key] && typeof source[key] === 'object' && !Array.isArray(source[key])) {
                if (!(key in target)) {
                    result[key] = source[key];
                } else {
                    result[key] = this.deepMerge(target[key], source[key]);
                }
            } else if (source[key] !== undefined) {
                result[key] = source[key];
            }
        });

        return result;
    }

    initialize() {
        if (!this.config.enabled) return;

        this.createStyles();
        this.createModal();
        this.attachEventListeners();

        // Initially hide the modal
        this.hideModal();

        // Check cookie status and show if needed
        const status = this.getStatus();
        if (!this.hasConsented()) {  // Changed to use hasConsented() check
            this.showModal();
        }
    }

    showModal(centered = false) {
        if (centered) {
            this.modal.classList.add('cc-center');
            this.overlay.style.display = 'block';
        }
        this.modal.classList.remove('cc-invisible');
    }

    hideModal() {
        this.modal.classList.add('cc-invisible');
        this.modal.classList.remove('cc-center');
        this.overlay.style.display = 'none';
    }

    // Add logging to cookie functions
    setStatus(status) {
        const { name, path, domain, expiryDays, secure } = this.config.cookie;
        const expires = new Date();
        expires.setDate(expires.getDate() + expiryDays);

        document.cookie = `${name}=${status};expires=${expires.toUTCString()};path=${path}${domain ? `;domain=${domain}` : ''}${secure ? ';secure' : ''}`;
    }

    getStatus() {
        const match = document.cookie.match(new RegExp(`${this.config.cookie.name}=([^;]+)`));
        const status = match ? match[1] : null;
        return status;
    }

    hasConsented() {
        const status = this.getStatus();
        return status === this.status.ALLOW;
    }

    submitPendingForm() {
        if (this.pendingForm) {
            // For Contact Form 7 forms
            if (this.pendingForm.classList.contains('wpcf7-form')) {
                const formWrapper = this.pendingForm.closest('.wpcf7');
                if (formWrapper) {
                    // Remove our submit event listener temporarily
                    document.removeEventListener('submit', this.handleFormSubmit);

                    // Let CF7 handle the submission
                    const submitEvent = new Event('submit', {
                        bubbles: true,
                        cancelable: true
                    });
                    this.pendingForm.dispatchEvent(submitEvent);

                    // Reattach our submit listener after a short delay
                    setTimeout(() => {
                        document.addEventListener('submit', (e) => this.handleFormSubmit(e));
                    }, 100);
                }
            } else {
                // For non-CF7 forms
                this.pendingForm.submit();
            }
            this.pendingForm = null;
        }
    }

    createStyles() {
        const styles = `
            .cc-window {
                background-color: ${this.config.palette.popup.background};
                color: ${this.config.palette.popup.text};
            }
            .cc-allow {
                background-color: ${this.config.palette.button.background};
                color: ${this.config.palette.button.text};
            }
            .cc-allow:hover {
                background-color: ${this.getLuminance(this.config.palette.button.background)};
            }
            .cc-deny {
                background-color: ${this.config.palette.highlight.background};
                color: ${this.config.palette.highlight.text};
                border-color: ${this.config.palette.highlight.border};
            }
            .cc-deny:hover {
                background-color: ${this.getLuminance(this.config.palette.highlight.background)};
            }
            .cc-link {
                color: ${this.config.palette.popup.text};
            }
        `;

        const styleElement = document.createElement('style');
        styleElement.textContent = styles;
        document.head.appendChild(styleElement);
    }

    getComplianceContent() {
        switch (this.config.compliance?.type) {
            case 'opt-out':
                return `
                    <button class="cc-btn cc-allow">${this.config.content.allow}</button>
                    ${this.config.content.deny ? `<button class="cc-btn cc-deny">${this.config.content.deny}</button>` : ''}
                `;
            case 'opt-in':
                return `
                    ${this.config.content.deny ? `<button class="cc-btn cc-deny">${this.config.content.deny}</button>` : ''}
                    <button class="cc-btn cc-allow">${this.config.content.allow}</button>
                `;
            case 'info':
                return `<button class="cc-btn cc-allow">${this.config.content.dismiss}</button>`;
            default:
                return `<button class="cc-btn cc-allow">${this.config.content.dismiss}</button>`;
        }
    }

    createModal() {
        const complianceContent = this.getComplianceContent();
        const modalHtml = `
            <div class="cc-window cc-theme-classic cc-invisible cc-bottom-left" role="dialog" aria-label="cookieconsent">
                <div class="cc-message">${this.config.content.message} ${this.config.content.link ? `
                    <a href="${this.config.content.href}" target="${this.config.content.target}" class="cc-link">${this.config.content.link}</a>
                ` : ''}</div>
                <div class="cc-compliance">
                    ${complianceContent}
                </div>
            </div>
            <div class="cc-overlay"></div>
        `;

        const modalElement = document.createElement('div');
        modalElement.innerHTML = modalHtml;
        document.body.appendChild(modalElement);

        this.modal = modalElement.querySelector('.cc-window');
        this.overlay = modalElement.querySelector('.cc-overlay');
    }

    attachEventListeners() {
        document.addEventListener('submit', (e) => this.handleFormSubmit(e));

        this.modal.querySelector('.cc-allow').addEventListener('click', () => {
            this.setStatus(this.status.ALLOW);
            this.hideModal();
            this.submitPendingForm();
        });

        if (this.modal.querySelector('.cc-deny')) {
            this.modal.querySelector('.cc-deny').addEventListener('click', () => {
                this.setStatus(this.status.DENY);
                this.hideModal();
            });
        }
    }

    // Add normaliseHex and getLuminance methods
    normaliseHex(hex) {
        if (hex[0] === '#') {
            hex = hex.substr(1);
        }
        if (hex.length === 3) {
            hex = hex[0] + hex[0] + hex[1] + hex[1] + hex[2] + hex[2];
        }
        return hex;
    }

    getLuminance(hex) {
        const color = this.normaliseHex(hex);
        const num = parseInt(color, 16);
        const amt = 38;

        const r = (num >> 16) + amt;
        const g = (num >> 8 & 0x00FF) + amt;
        const b = (num & 0x0000FF) + amt;

        return '#' + (0x1000000 +
            (r < 255 ? (r < 1 ? 0 : r) : 255) * 0x10000 +
            (g < 255 ? (g < 1 ? 0 : g) : 255) * 0x100 +
            (b < 255 ? (b < 1 ? 0 : b) : 255))
            .toString(16)
            .slice(1);
    }

    // Move adjustColor method inside the class
    adjustColor(color, percent) {
        const num = parseInt(color.replace('#', ''), 16),
            amt = Math.round(2.55 * percent),
            R = (num >> 16) + amt,
            G = (num >> 8 & 0x00FF) + amt,
            B = (num & 0x0000FF) + amt;
        return '#' + (0x1000000 + (R < 255 ? R < 1 ? 0 : R : 255) * 0x10000 +
            (G < 255 ? G < 1 ? 0 : G : 255) * 0x100 +
            (B < 255 ? B < 1 ? 0 : B : 255)).toString(16).slice(1);
    }
}

// Initialize when the DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    if (window.cookieconsent_config) {
        window.cookieconsent = new CookieConsent(window.cookieconsent_config);
    }
});