<?php
/*
Plugin Name: Klantenvertellen
Plugin URI:  http://www.yndenz.com
Description: Klantenvertellen.nl page reader for reviews
Version:     1.2.6
Author:      Gido Manders <g.manders@yndenz.com>
Author URI:  http://www.yndenz.com
Domain Path: /languages
Text Domain: klantenvertellen
*/

/**
 * Add a menu item to the settings menu
 */
add_action('admin_menu', 'klantenvertellen_menu');
function klantenvertellen_menu()
{
    // Add a new submenu under Settings
    add_options_page(__('Klantenvertellen', 'klantenvertellen'), __('Klantenvertellen', 'klantenvertellen'), 'manage_options', 'klantenvertellen', 'klantenvertellen_settings_page');
}

/**
 * Display the settings page
 */
function klantenvertellen_settings_page()
{
    //must check that the user has the required capability
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.'));
    }

    // See if the user has saved the settings
    if (isset($_POST['klantenvertellen_slug'])) {
        // Save the posted value in the database
        update_option('klantenvertellen_slug', $_POST['klantenvertellen_slug']);
        ?>
        <div class="updated"><p><strong><?php _e('Settings saved.', 'klantenvertellen'); ?></strong></p></div>
        <?php
    }
    ?>
    <div class="wrap">
        <h2><?php _e('Klantenvertellen Settings', 'klantenvertellen'); ?></h2>
        <form name="klantenvertellenSettingsForm" method="post" action="">
            <label>
                <?php _e("Slug:", 'klantenvertellen'); ?>
            </label>
            <input type="text" name="klantenvertellen_slug" value="<?php echo get_option('klantenvertellen_slug'); ?>"/>
            <button type="submit" class="button-primary"><?php _e('Save', 'klantenvertellen'); ?></button>
        </form>
    </div>
    <?php
}

/**
 * Get all klantenvertellen reviews.
 * Define KLANTENVERTELLEN_CACHE if you want to cache the reviews (updated hourly)
 *
 * @return object|boolean Returns an array containing the reviews, or false if an error occurred.
 */
function klantenvertellen_get_reviews($cron = false)
{
    // If the cache file is not defined or the file is updated over an hour ago, fetch the reviews
    if ($cron && !defined('KLANTENVERTELLEN_CACHE') || $cron && !file_exists(KLANTENVERTELLEN_CACHE) || $cron && filemtime(KLANTENVERTELLEN_CACHE) < strtotime('-1 hour')) {

        $base_url = 'https://mobiliteit.klantenvertellen.nl/beoordelingen/' . get_option('klantenvertellen_slug');
        $base_url = apply_filters('klantenvertellen_base_url', $base_url);
        try {
            // Ignore invalid HTML warnings
            libxml_use_internal_errors(true);

            $dom = new DomDocument();
            @$dom->loadHTMLFile($base_url);
            $xpath = new DOMXPath($dom);

            // Get last page available
            $last_page = 0;
            if ($xpath->query('//*[@class="pagination"]')->length) {
                $last_page = (int)$xpath->query('//*[@class="pagination"]/a[last()-1]/text()')->item(0)->nodeValue - 1;
            }

            $result = new stdClass();
            // Set the average
            $average = $xpath->query('//*[contains(@class,"statistics")]//*[contains(@class,"main")]//*[contains(@class,"score")]/*[contains(@class,"rating")]/text()');
            $average_as_string = trim((string)$average->item(0)->nodeValue);
            $result->average = (float)str_replace(',', '.', $average_as_string);
            $result->num_reviews = $xpath->query('//*[contains(@class,"statistics")]//*[contains(@class,"main")]//*[contains(@class,"score")]/*[contains(@class,"numberofreviews")]/*[contains(@class,"count")]/text()')->item(0)->nodeValue;
            $result->num_reviews = number_format($result->num_reviews,0,',','.');
            $result->percentage_recommending = substr(trim($xpath->query('//*[contains(@class,"statistics")]//*[contains(@class,"main")]//*[contains(@class,"footer")]/*[contains(@class,"recommendation")]/text()')->item(0)->nodeValue), 0, 2);


            // Check whether the number of reviews in cache file and remote location match
            // It is highly probable that an update is not needed if these values match
            // So if they do we return the cached result immediately
            if(file_exists(KLANTENVERTELLEN_CACHE)){
                $cached_result = json_decode(file_get_contents(KLANTENVERTELLEN_CACHE));
                if($cached_result->num_reviews == $result->num_reviews)
                    return $cached_result;
            }

            // Loop the pages and convert the reviews
            $result->reviews = [];
            for ($i = 0; $i <= $last_page; $i++) {
                if ($i == 0) {
                    $page = $dom;
                } else {
                    $page = new DomDocument();
                    @$page->loadHTMLFile($base_url . '/' . $i);
                }
                $page_xpath = new DOMXPath($page);

                // Loop the reviews to convert them into a readable item
                $reviews = $page_xpath->query('//*[contains(@class,"reviews")]//*[@class="review"]');
                foreach ($reviews as $review) {
                    $rating = $page_xpath->query('.//*[contains(@class ,"score")]/*[contains(@class ,"rating")]/text()', $review);
                    $rating_as_string = trim((string)$rating->item(0)->nodeValue);
                    if ($rating->length === 0 || $rating_as_string === 'n.v.t.') {
                        continue;
                    }
                    $item = new stdClass();
                    $item->author = trim((string)$page_xpath->query('.//*[contains(@class,"title")]/h2/text()', $review)->item(0)->nodeValue);
                    $date = trim((string)$page_xpath->query('.//*[contains(@class,"title")]/*[contains(@class,"date")]/text()', $review)->item(0)->nodeValue);
                    $item->date = implode('-',array_reverse(explode('-',$date)));
                    $item->rating = (float)str_replace(',', '.', $rating_as_string);
                    $item->recommend = strpos((string)$page_xpath->query('.//*[contains(@class,"recommendation")]/text()', $review)->item(0)->nodeValue, 'Ja') > 0;
                    $item->reason = trim((string)$page_xpath->query('.//*[contains(@class,"answer")]/p[1]/text()', $review)->item(0)->nodeValue);
                    $description = $page_xpath->query('.//*[contains(@class,"answer")]/p[not(./following-sibling::p)]/text()', $review);
                    $item->text = '';
                    if ($description->length > 0) {
                        foreach ($description as $child) {
                            $item->text .= $child->nodeValue;
                        }
                    }
                    if (isset($cached_result)) {
                        // if the loop encounters a review that already exists
                        // every review after that is probably already in the cached
                        // This is an exact object match
                        foreach ($cached_result->reviews as $creview):
                            if($item == $creview)
                                break 3;
                        endforeach;
                    }
                    $result->reviews[] = $item;
                }
            }

            if (isset($cached_result)) {
                $result->reviews = array_merge($result->reviews,$cached_result->reviews);
            }
            
            $result->found_reviews = count($result->reviews);

            // If the cache file is defined, save the results JSON encoded in the file
            if ($result->found_reviews > 0) {
                if (defined('KLANTENVERTELLEN_CACHE')) {
                    file_put_contents(KLANTENVERTELLEN_CACHE, json_encode($result));
                }
            } else {
                mail('test@yndenz.com',get_bloginfo('title').' 0 Reviews',$_SERVER['HTTP_HOST']);
                throw new Exception(__('No reviews found', 'klantenvertellen'));
            }

            ob_start();
	        include locate_template('klantenvertellen-item-reviewed.php');
	        $result->item_reviewed = ob_get_clean();

            return $result;
        } catch (Exception $e) {
            if (WP_DEBUG) {
                var_dump($e);
            }

            // If the cache file is defined and the file exists, return the cached results to prevent empty results
            // This could help in case klantenvertellen.nl is down or an error occurred while reading a page
            if (defined('KLANTENVERTELLEN_CACHE') && file_exists(KLANTENVERTELLEN_CACHE)) {
                $result = json_decode(file_get_contents(KLANTENVERTELLEN_CACHE));

	            ob_start();
	            include locate_template('klantenvertellen-item-reviewed.php');
	            $result->item_reviewed = ob_get_clean();

	            return $result;
            }

            // An error occurred, so we return false
            return false;
        }
    } else {
        // Read the cache file and decode the contents to return the results as an object
        $result = json_decode(file_get_contents(KLANTENVERTELLEN_CACHE));

	    ob_start();
	    include locate_template('klantenvertellen-item-reviewed.php');
	    $result->item_reviewed = ob_get_clean();

	    return $result;
    }
}

/**
 * Get average klantenvertellen.nl rating
 */
add_shortcode('klantvertellen_average', 'klantvertellen_average');
function klantvertellen_average()
{
    $klantvertellen = klantenvertellen_get_reviews();
    return $klantvertellen->average;
}

/**
 * Get total klantenvertellen.nl reviews
 */
add_shortcode('klantvertellen_total', 'klantvertellen_total');
function klantvertellen_total()
{
    $klantvertellen = klantenvertellen_get_reviews();
    return $klantvertellen->found_reviews;
}
