<?php

namespace yndenz\Plugins\LazyLoader;

use Spatie\ImageOptimizer\OptimizerChainFactory;
use Spatie\ImageOptimizer\Optimizers\Jpegoptim;
use Spatie\ImageOptimizer\Optimizers\Optipng;
use Spatie\ImageOptimizer\Optimizers\Pngquant;

class LazyLoadImages {

	public static function init() {
		add_filter( 'wp_enqueue_scripts', array( static::class, 'enqueueScripts' ) );
		add_filter( 'wp_get_attachment_image_attributes', array( static::class, 'prepareImage' ), 999 );
	}

	/**
	 * Enqueue the scripts used for lazy loading images.
	 */
	public static function enqueueScripts() {
		wp_enqueue_script('lazyload-images', plugin_dir_url( __FILE__ ) . 'lazyload-images.min.js', array(), false, true );
	}

	/**
	 * Prepare an attachment image for lazy loading.
	 *
	 * @param array $attr
	 *
	 * @return array
	 */
	public static function prepareImage( $attr ) {
		if ( is_admin() ) {
			return $attr;
		}

		$upload_dir = _wp_upload_dir();

		$src_path = str_replace( $upload_dir['baseurl'], $upload_dir['basedir'], $attr['src'] );

		if ( ! file_exists( $src_path ) ) {
			return $attr;
		}

		$lazy_image = static::getLowQualityImage( $src_path );

		if ( $lazy_image === false ) {
			return $attr;
		}

		$attr['data-src'] = $attr['src'];
		$attr['src']      = str_replace( $upload_dir['basedir'], $upload_dir['baseurl'], $lazy_image );

		if ( array_key_exists( 'srcset', $attr ) ) {
			$attr['data-srcset'] = $attr['srcset'];
			unset( $attr['srcset'] );
		}

		if ( array_key_exists( 'class', $attr ) ) {
			$attr['class'] .= ' ym-lazy';
		} else {
			$attr['class'] = 'ym-lazy';
		}

		return $attr;
	}

	protected static function getLowQualityImage( $file ) {
		$pathinfo = pathinfo( $file );

		if ( ! in_array( strtolower( $pathinfo['extension'] ), array( 'jpg', 'jpeg', 'png' ) ) ) {
			return false;
		}

		$lazy_image = $pathinfo['dirname'] . '/' . $pathinfo['filename'] . '-lazy.' . $pathinfo['extension'];

		if ( ! file_exists( $lazy_image ) ) {
			static::optimizeLazyImage( $file, $lazy_image, $pathinfo['extension'] );
		}

		return $lazy_image;
	}

	/**
	 * Create a heavily blurred and optimized version of an image.
	 *
	 * @param string $src
	 * @param string $dest
	 * @param string $ext
	 */
	protected static function optimizeLazyImage( $src, $dest, $ext ) {
		static::blurImage( $src, $dest, $ext );

		$optimizerChain = OptimizerChainFactory::create();
		if ( $ext === 'png' ) {
			$optimizerChain->addOptimizer( new Pngquant( array( '--quality=0-10' ) ) );
			$optimizerChain->addOptimizer( new Optipng( array( '-o2 -strip all' ) ) );
		} else {
			$optimizerChain->addOptimizer( new Jpegoptim( array( '-m10', '--strip-all', '--all-progressive' ) ) );
		}
		$optimizerChain->optimize( $dest );
	}

	/**
	 * Create a heavily blurred version of an image.
	 *
	 * @param string $src
	 * @param string $dest
	 * @param string $ext
	 */
	protected static function blurImage( $src, $dest, $ext ) {
		if ( $ext === 'png' ) {
			$image = imagecreatefrompng( $src );
		} else {
			$image = imagecreatefromjpeg( $src );
		}
		list( $w, $h ) = getimagesize( $src );

		// Define scale for downsized image
		$size = array(
			'sm' => array( 'w' => intval( $w / 8 ), 'h' => intval( $h / 8 ) ),
			'md' => array( 'w' => intval( $w / 2 ), 'h' => intval( $h / 2 ) )
		);

		// Scale by 12.5% and apply Gaussian blur
		$sm = static::resizeImage( $image, $ext, $w, $h, $size['sm']['w'], $size['sm']['h'] );
		imagedestroy( $image );
		static::filterImage( $sm );

		// Scale result by 400% and blur again
		$md = static::resizeImage( $sm, $ext, $size['sm']['w'], $size['sm']['h'], $size['md']['w'], $size['md']['h'] );
		imagedestroy( $sm );
		static::filterImage( $md, 25 );

		// Scale result back to original size and blur again
		$image = static::resizeImage( $md, $ext, $size['md']['w'], $size['md']['h'], $w, $h );
		imagedestroy( $md );
		static::filterImage( $image, 1, null );

		// Store the blurred image permanently
		if ( $ext === 'png' ) {
			imagepng( $image, $dest );
		} else {
			imagejpeg( $image, $dest );
		}
		imagedestroy( $image );
	}

	protected static function resizeImage( &$image, $ext, $w, $h, $new_w, $new_h ) {
		$new_image = imagecreatetruecolor( $new_w, $new_h );

		if ( $ext === 'png' ) {
			imagealphablending( $new_image, false );
		}

		imagecopyresampled( $new_image, $image, 0, 0, 0, 0, $new_w, $new_h, $w, $h );

		if ( $ext === 'png' ) {
			imagesavealpha( $new_image, true );
		}

		return $new_image;
	}

	/**
	 * Blur the image.
	 *
	 * @param resource $image
	 * @param int      $repeat Number of times to repeat blur filter.
	 * @param int      $brightness
	 */
	protected static function filterImage( &$image, $repeat = 40, $brightness = 999 ) {
		for ( $x = 0; $x < $repeat; $x ++ ) {
			imagefilter( $image, IMG_FILTER_GAUSSIAN_BLUR, $brightness );
		}
		imagefilter( $image, IMG_FILTER_SMOOTH, 99 );
		imagefilter( $image, IMG_FILTER_BRIGHTNESS, 10 );
	}

}
