<?php

namespace yndenz\Plugins\LazyLoader;

use Exception;
use WP_Post;

class LazyLoadImages {

	public static function init() {
		add_filter( 'wp_enqueue_scripts', array( static::class, 'enqueueScripts' ) );
		add_filter( 'wp_get_attachment_image_attributes', array( static::class, 'prepareImage' ), 999 );
		add_filter( 'wp_generate_attachment_metadata', array( static::class, 'afterUpload' ), 20, 2 );
		add_action( 'lazy_load_prepare_attachment', array( static::class, 'prepareAttachment' ) );
	}

	/**
	 * Enqueue the scripts used for lazy loading images.
	 */
	public static function enqueueScripts() {
		wp_enqueue_script( 'lazyload-images', plugin_dir_url( __FILE__ ) . 'lazyload-images.min.js', array(), false, true );
	}

	/**
	 * Prepare an attachment image for lazy loading.
	 *
	 * @param array $attr
	 *
	 * @return array
	 */
	public static function prepareImage( $attr ) {
		if ( is_admin() ) {
			return $attr;
		}

		$upload_dir = _wp_upload_dir();

		$src_path = str_replace( $upload_dir['baseurl'], $upload_dir['basedir'], $attr['src'] );

		if ( ! file_exists( $src_path ) ) {
			return $attr;
		}

		try {
			$image      = new LazyImage( $src_path );
			$lazy_image = $image->createLazyImage();
		} catch ( Exception $e ) {
			return $attr;
		}

		$attr['data-src'] = $attr['src'];
		$attr['src']      = str_replace( $upload_dir['basedir'], $upload_dir['baseurl'], $lazy_image );

		if ( array_key_exists( 'srcset', $attr ) ) {
			$attr['data-srcset'] = $attr['srcset'];
			unset( $attr['srcset'] );
		}

		if ( array_key_exists( 'class', $attr ) ) {
			$attr['class'] .= ' ym-lazy';
		} else {
			$attr['class'] = 'ym-lazy';
		}

		return $attr;
	}

	/**
	 * Prepare lazy loading images using a cronjob after an image is uploaded.
	 *
	 * @param array $metadata
	 * @param int   $attachment_id
	 *
	 * @return array
	 */
	public static function afterUpload( $metadata, $attachment_id ) {
		wp_schedule_single_event( time(), 'lazy_load_prepare_attachment', array( 'attachment_id' => $attachment_id ) );

		return $metadata;
	}

	/**
	 * Create lazy loading images for an attachment.
	 *
	 * @param int $attachment_id
	 *
	 * @throws Exception
	 */
	public static function prepareAttachment( $attachment_id ) {
		$attachment = get_post( $attachment_id );
		if ( ! $attachment instanceof WP_Post ) {
			throw new Exception( 'Image not found' );
		}

		foreach ( get_intermediate_image_sizes() as $size ) {
			try {
				$image = new LazyImage( self::_getAttachmentFilePath( $attachment->ID, $size ) );
				$image->createLazyImage();
			} catch ( Exception $e ) {
				// Do nothing
			}
		}
	}

	/**
	 * Get the absolute file path of an attachment.
	 *
	 * @param int    $attachment_id
	 * @param string $size
	 *
	 * @return string
	 */
	private static function _getAttachmentFilePath( $attachment_id, $size = 'full' ) {
		$upload_dir = wp_upload_dir();

		return str_replace( $upload_dir['baseurl'], $upload_dir['basedir'], wp_get_attachment_image_src( $attachment_id, $size )[0] );
	}

}
