<?php

namespace yndenz\Plugins\LazyLoader;

use Exception;
use Spatie\ImageOptimizer\OptimizerChainFactory;
use Spatie\ImageOptimizer\Optimizers\Jpegoptim;
use Spatie\ImageOptimizer\Optimizers\Optipng;
use Spatie\ImageOptimizer\Optimizers\Pngquant;

class LazyImage {

	public $path;

	protected $_path_info;

	/**
	 * LazyImage constructor.
	 *
	 * @param string $path
	 *
	 * @throws Exception
	 */
	public function __construct( $path ) {
		if ( ! file_exists( $path ) ) {
			throw new Exception( 'File not found' );
		}

		$this->path       = $path;
		$this->_path_info = pathinfo( $path );

		if ( ! in_array( strtolower( $this->_path_info['extension'] ), array( 'jpg', 'jpeg', 'png' ) ) ) {
			throw new Exception( 'File type not supported' );
		}
	}

	/**
	 * Create a heavily blurred and optimized version of the image.
	 *
	 * @return string
	 */
	public function createLazyImage() {
		$lazy_file_path = $this->getLazyFilePath();
		if ( file_exists( $lazy_file_path ) ) {
			return $lazy_file_path;
		}

		self::_blur();

		$optimizerChain = OptimizerChainFactory::create();
		if ( $this->_path_info['extension'] === 'png' ) {
			$optimizerChain->addOptimizer( new Pngquant( array( '--quality=0-10', '--force', '--strip' ) ) );
			$optimizerChain->addOptimizer( new Optipng( array( '-o2 -strip all' ) ) );
		} else {
			$optimizerChain->addOptimizer( new Jpegoptim( array( '-m10', '--strip-all', '--all-progressive' ) ) );
		}
		$optimizerChain->optimize( $lazy_file_path );

		return $lazy_file_path;
	}

	/**
	 * Create a heavily blurred version of the image.
	 */
	protected function _blur() {
		if ( $this->_path_info['extension'] === 'png' ) {
			$image = imagecreatefrompng( $this->path );
		} else {
			$image = imagecreatefromjpeg( $this->path );
		}
		list( $w, $h ) = getimagesize( $this->path );

		// Define scale for downsized image
		$size = array(
			'sm' => array( 'w' => intval( $w / 8 ), 'h' => intval( $h / 8 ) ),
			'md' => array( 'w' => intval( $w / 2 ), 'h' => intval( $h / 2 ) )
		);

		// Scale by 12.5% and apply Gaussian blur
		$sm = self::_resize( $image, $this->_path_info['extension'], $w, $h, $size['sm']['w'], $size['sm']['h'] );
		imagedestroy( $image );
		self::_filter( $sm );

		// Scale result by 400% and blur again
		$md = self::_resize( $sm, $this->_path_info['extension'], $size['sm']['w'], $size['sm']['h'], $size['md']['w'], $size['md']['h'] );
		imagedestroy( $sm );
		self::_filter( $md, 25 );

		// Scale result back to original size and blur again
		$image = self::_resize( $md, $this->_path_info['extension'], $size['md']['w'], $size['md']['h'], $w, $h );
		imagedestroy( $md );
		self::_filter( $image, 1, null );

		// Store the blurred image permanently
		if ( $this->_path_info['extension'] === 'png' ) {
			imagepng( $image, $this->getLazyFilePath() );
		} else {
			imagejpeg( $image, $this->getLazyFilePath() );
		}
		imagedestroy( $image );
	}

	/**
	 * Resize the image.
	 *
	 * @param resource $image
	 * @param string   $ext
	 * @param int      $w
	 * @param int      $h
	 * @param int      $new_w
	 * @param int      $new_h
	 *
	 * @return false|resource
	 */
	protected static function _resize( &$image, $ext, $w, $h, $new_w, $new_h ) {
		$new_image = imagecreatetruecolor( $new_w, $new_h );

		if ( $ext === 'png' ) {
			imagealphablending( $new_image, false );
		}

		imagecopyresampled( $new_image, $image, 0, 0, 0, 0, $new_w, $new_h, $w, $h );

		if ( $ext === 'png' ) {
			imagesavealpha( $new_image, true );
		}

		return $new_image;
	}

	/**
	 * Blur the image.
	 *
	 * @param resource $image
	 * @param int      $repeat Number of times to repeat blur filter.
	 * @param int      $brightness
	 */
	protected static function _filter( &$image, $repeat = 40, $brightness = 999 ) {
		for ( $x = 0; $x < $repeat; $x ++ ) {
			imagefilter( $image, IMG_FILTER_GAUSSIAN_BLUR, $brightness );
		}
		imagefilter( $image, IMG_FILTER_SMOOTH, 99 );
		imagefilter( $image, IMG_FILTER_BRIGHTNESS, 10 );
	}

	/**
	 * Get file path for the lazy version of the image.
	 *
	 * @return string
	 */
	public function getLazyFilePath() {
		$lazy_file_path = $this->_path_info['dirname'] . '/' . $this->_path_info['filename'] . '-lazy.' . $this->_path_info['extension'];

		return apply_filters( 'yndenz_lazy_loader_lazy_file_path', $lazy_file_path, $this->path );
	}

}
