<?php

namespace yndenz\Modules\NoIndexSitemap;

use Exception;
use WP_Query;
use WPSEO_Meta;
use WPSEO_Sitemaps;
use WPSEO_Taxonomy_Meta;
use yndenz\Modules\BaseModule\BaseModule;

class NoIndexSitemap extends BaseModule {
	public static $name = 'noindex';

	/**
	 * Register a no-index sitemap for W3 Total Cache to prime those pages as well.
	 */
	public static function register() {
		/** @var WPSEO_Sitemaps $wpseo_sitemaps */
		global $wpseo_sitemaps;
		$wpseo_sitemaps->register_sitemap( self::$name, array( static::class, 'build' ) );
	}

	/**
	 * Build a no-index sitemap for W3 Total Cache to prime those pages as well.
	 */
	public static function build() {
		/** @var WPSEO_Sitemaps $wpseo_sitemaps */
		global $wpseo_sitemaps;
		$links   = self::_getUrls();
		$sitemap = $wpseo_sitemaps->renderer->get_sitemap( $links, self::$name, get_query_var( 'sitemap_n' ) );
		$wpseo_sitemaps->set_sitemap( $sitemap );
	}

	/**
	 * Get all no-index URLs for the no-index sitemap.
	 *
	 * @return array
	 */
	private static function _getUrls() {
		global $post;
		$urls    = array();
		$lastmod = date( 'c' );

		try {
			$post_types = array_map( function ( $post_type ) {
				return $post_type->name;
			}, array_filter( get_post_types( array(), 'objects' ), function ( $post_type ) {
				return $post_type->public;
			} ) );

			$query = new WP_Query( array( 'post_type' => $post_types, 'posts_per_page' => - 1 ) );
			while ( $query->have_posts() ) {
				$query->the_post();
				if ( WPSEO_Meta::get_value( 'meta-robots-noindex', $post->ID ) !== '1' ) {
					continue;
				}
				array_push( $urls, array(
					'loc'    => get_permalink( $post ),
					'mod'    => $lastmod,
					'chf'    => 'daily',
					'pri'    => 1,
					'images' => array()
				) );
			}

			$taxonomies = array_map( function ( $taxonomy ) {
				return $taxonomy->name;
			}, array_filter( get_taxonomies( array(), 'objects' ), function ( $taxonomy ) {
				return $taxonomy->public;
			} ) );

			$terms = get_terms( array( 'hide_empty' => false, 'taxonomy' => $taxonomies ) );
			foreach ( $terms as $term ) {
				if ( WPSEO_Taxonomy_Meta::get_term_meta( $term, $term->taxonomy, 'noindex' ) === 'noindex' ) {
					continue;
				}
				array_push( $urls, array(
					'loc'    => get_term_link( $term, $term->taxonomy ),
					'mod'    => $lastmod,
					'chf'    => 'daily',
					'pri'    => 1,
					'images' => array()
				) );
			}
		} catch ( Exception $e ) {
			$name = self::$name;
			error_log( "Unable to build {$name} sitemap: {$e->getMessage()}" );
		}

		return $urls;
	}
}
