<?php

namespace yndenz\Plugins\NotifyYndenz;

use Exception;

class ConsoleError {

	private static $_email_subject = 'Failed to send this console error to yndenz Notify';

	/**
	 * Create a database table to store console errors before they're sent in batches.
	 */
	public static function create_database_table() {
		global $wpdb;

		$table_name      = $wpdb->prefix . 'console_errors';
		$charset_collate = $wpdb->get_charset_collate();

		$sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
		    id bigint(20) NOT NULL AUTO_INCREMENT,
		    browser varchar(255) NOT NULL,
		    url varchar(255) NOT NULL,
		    type varchar(255) NOT NULL,
		    message text NULL,
		    ip varchar(50) NOT NULL,
		    created_at timestamp DEFAULT CURRENT_TIMESTAMP NOT NULL,
		    PRIMARY KEY  (id)
		) {$charset_collate}";

		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
		dbDelta( $sql );
	}

	/**
	 * Enqueue script to catch console errors.
	 */
	public static function enqueueScripts() {
		wp_enqueue_script( 'notify-yndenz-console-errors', plugin_dir_url( __FILE__ ) . 'console-errors.min.js' );
		$ajax_url = admin_url( 'admin-ajax.php' );
		wp_add_inline_script( 'notify-yndenz-console-errors', "window.ajax_url = '{$ajax_url}';", 'before' );
	}

	/**
	 * Catch console errors to send them in a batch to yndenz.
	 */
	public static function store() {
		$bots = self::_fetchBots();

		if ( preg_match( '/' . implode( '|', $bots['user_agents'] ) . '/i', $_SERVER['HTTP_USER_AGENT'] ) ) {
			exit;
		}

		$domain = gethostbyaddr( $_SERVER['REMOTE_ADDR'] );
		if ( preg_match( '/' . implode( '|', $bots['hosts'] ) . '/i', $domain ) ) {
			exit;
		}

		global $wpdb;

		$wpdb->insert( $wpdb->prefix . 'console_errors', array(
			'browser' => $_SERVER['HTTP_USER_AGENT'],
			'url'     => array_key_exists( 'HTTP_REFERER', $_SERVER ) ? $_SERVER['HTTP_REFERER'] : 'undefined',
			'type'    => esc_sql( $_POST['type'] ),
			'message' => esc_sql( $_POST['text'] ),
			'ip'      => $_SERVER['REMOTE_ADDR']
		) );

		exit;
	}

	/**
	 * Fetch bots configuration.
	 * Caches the bots file locally for 5 minutes.
	 * Fetches the bots file from the yndenz server if necessary.
	 */
	private static function _fetchBots() {
		$upload_dir = wp_upload_dir();
		$file       = $upload_dir['basedir'] . '/bots.json';

		if ( ! file_exists( $file ) || filemtime( $file ) < strtotime( '-5 minutes' ) ) {
			$ch = curl_init( 'https://gitlab.yndenz.com/api/v4/snippets/70/raw' );
			curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
			curl_setopt( $ch, CURLOPT_HTTPHEADER, array(
				'PRIVATE-TOKEN: i3iqFGjAwCHXNTiYJGX8'
			) );
			$contents = curl_exec( $ch );
			if ( ! empty( $contents ) ) {
				file_put_contents( $file, $contents );
			}
			curl_close( $ch );
		} else {
			$contents = file_get_contents( $file );
		}

		$bots = json_decode( $contents, true );

		return empty( $bots ) ? array( 'user_agents' => array(), 'hosts' => array() ) : $bots;
	}

	/**
	 * Schedule a cron event to send console errors grouped to yndenz notify.
	 */
	public static function schedule_check() {
		if ( wp_next_scheduled( 'check_console_errors' ) === false ) {
			wp_schedule_single_event( time(), 'check_console_errors' );
		}

		add_action( 'check_console_errors', array( self::class, 'check' ) );
	}

	/**
	 * Check if there are any console errors that have to be sent to yndenz.
	 * Notify yndenz if the last time a console error for that IP was logged is over 10 seconds ago.
	 *
	 * @throws Exception
	 */
	public static function check() {
		$errors = self::get_console_errors();
		if ( $errors === false ) {
			return;
		}

		global $wpdb;

		foreach ( self::map_by_property( $errors, 'ip' ) as $ip => $errors_by_ip ) {
			foreach ( self::map_by_property( $errors_by_ip, 'url' ) as $url => $errors_by_url ) {
				foreach ( self::map_by_property( $errors_by_ip, 'type' ) as $type => $console_errors ) {
					$created_at = strtotime( array_values( $console_errors )[0]['created_at'] );

					$message = array(
						'username'    => get_bloginfo( 'name' ),
						'text'        => $type . ' (' . count( $console_errors ) . ') on ' . $url . ' for ' . $ip,
						'attachments' => array(
							array(
								'title' => date( 'Y-m-d H:i:s', $created_at ),
								'text'  => implode( "\r\n", array_map( function ( $error ) {
										return $error['message'];
									}, $console_errors ) ) . "\r\n" . $console_errors[0]['browser']
							)
						)
					);

					if ( Message::send( $message, $created_at > strtotime( '-1 hour' ), static::$_email_subject ) ) {
						foreach ( $console_errors as $error ) {
							$wpdb->delete( $wpdb->prefix . 'console_errors', array( 'id' => $error['id'] ) );
						}
					}
				}
			}
		}
	}

	/**
	 * Get all console errors that are eligible to notify.
	 *
	 * @return array|bool
	 * @throws Exception
	 */
	private static function get_console_errors() {
		global $wpdb;

		$result = $wpdb->get_col( "SELECT c1.ip FROM {$wpdb->prefix}console_errors c1 JOIN (SELECT ip, url, max(created_at) as created_at FROM {$wpdb->prefix}console_errors GROUP BY ip, url) AS c2 ON c1.created_at = c2.created_at AND c1.ip = c2.ip AND c1.url = c2.url WHERE c1.created_at <= NOW() - INTERVAL 10 SECOND" );
		if ( empty( $result ) ) {
			return false;
		}

		$errors = $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}console_errors WHERE ip IN ('" . implode( "','", $result ) . "') ORDER BY created_at ASC", ARRAY_A );
		if ( empty( $errors ) ) {
			throw new Exception( 'Failed to fetch console errors by IP' );
		}

		return $errors;
	}

	/**
	 * Map console errors by IP.
	 *
	 * @param array  $errors
	 * @param string $property
	 *
	 * @return array
	 */
	private static function map_by_property( $errors, $property ) {
		$console_errors = array();
		foreach ( $errors as $error ) {
			if ( array_key_exists( $error[ $property ], $console_errors ) ) {
				array_push( $console_errors[ $error[ $property ] ], $error );
			} else {
				$console_errors[ $error[ $property ] ] = array( $error );
			}
		}

		return $console_errors;
	}

}
