<?php

namespace yndenz\Plugins\NotifyYndenz;

class PHPError {

	private static $_email_subject = 'Failed to send this PHP error to yndenz Notify';

	/**
	 *  Function to add the php error table to the database
	 */
	public static function create_database_table() {
		global $wpdb;

		$table_name      = $wpdb->prefix . 'php_errors';
		$charset_collate = $wpdb->get_charset_collate();

		$sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
		    id bigint(20) NOT NULL AUTO_INCREMENT,
		    error_message text NULL,
		    created_at timestamp NULL,
		    PRIMARY KEY  (id)
		) {$charset_collate}";

		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
		dbDelta( $sql );
	}

	/**
	 * Notify yndenz about PHP errors.
	 *
	 * @param int    $errno
	 * @param string $errstr
	 *
	 * @return bool
	 */
	public static function error_handler( $errno, $errstr ) {
		if ( error_reporting() === 0 ) {
			return true;
		}

		$message = array(
			'username'    => get_bloginfo( 'name' ),
			'text'        => self::_addRequestDetails( $errstr ),
			'attachments' => array(
				array(
					'title' => date( 'Y-m-d H:i:s' ),
					'text'  => self::log_trace( $errstr ),
				)
			)
		);

		Message::send( $message, true, static::$_email_subject, self::_getNotifyHookUrl(), $errstr );

		return true;
	}

	/**
	 * Get the debug backtrace as a string.
	 *
	 * @return string
	 */
	private static function log_trace( $error ) {
		$trace        = array_slice( debug_backtrace(), 1 );
		$call         = array_shift( $trace );
		$call['file'] = ! empty( $call['file'] ) ? $call['file'] : '-';
		$call['line'] = ! empty( $call['line'] ) ? $call['line'] : '-';
		$message      = $error . ' in ' . $call['file'] . ' on line ' . $call['line'];

		foreach ( $trace as $entry_id => $entry ) {
			$entry['file'] = ! empty( $entry['file'] ) ? $entry['file'] : '-';
			$entry['line'] = ! empty( $entry['line'] ) ? $entry['line'] : '-';
			if ( empty( $entry['class'] ) ) {
				$message .= sprintf( "\r\n" . '%3s. %s() %s:%s', $entry_id + 1, $entry['function'], $entry['file'], $entry['line'] );
			} else {
				$message .= sprintf( "\r\n" . '%3s. %s%s%s() %s:%s', $entry_id + 1, $entry['class'], $entry['type'], $entry['function'], $entry['file'], $entry['line'] );
			}
		}

		return $message;
	}

	/**
	 * Notify yndenz about PHP fatal errors.
	 */
	public static function notify_shutdown() {
		$error = error_get_last();
		if ( is_null( $error ) ) {
			return;
		}

		$error_message = $error['message'];
		if ( strpos( $error_message, "\n" ) !== false ) {
			$error_message = strstr( $error_message, "\n", true );
		}

		$message = array(
			'username'    => get_bloginfo( 'name' ),
			'text'        => self::_addRequestDetails( $error_message ),
			'attachments' => array(
				array(
					'title' => date( 'Y-m-d H:i:s' ),
					'text'  => $error_message !== $error['message'] ? substr( strstr( $error['message'], "\n" ), 1 ) : $error_message,
				)
			)
		);

		Message::send( $message, true, static::$_email_subject, self::_getNotifyHookUrl(), $error_message );
	}

	/**
	 * Remove the past error messages as the timestamp has been reached
	 *
	 * @param string $plainErrorMessage
	 *
	 * @return bool
	 */
	public static function removePastErrorMessages($plainErrorMessage) {
		global $wpdb;

		return $wpdb->delete("{$wpdb->prefix}php_errors", array("error_message" => $plainErrorMessage));
	}

	/**
	 * Check when the error message has last been sent.
	 *
	 * @param string $plainErrorMessage
	 * @param string $time
	 *
	 * @return bool
	 */
	public static function _checkErrorAlreadySent( $plainErrorMessage, $time ) {
		global $wpdb;

		// Get the current error time minus 1 hour, this to see if the 1 hour mark is passed when comparing with a found database entry
		$timeToCheck = date('Y-m-d H:i:s', strtotime($time . '-1 hour'));

		// Escape the single quote in the error message
		$formattedPlainErrorMessage = str_replace("'", "''", $plainErrorMessage);

		$foundErrorMessage = $wpdb->get_results("SELECT * FROM ". $wpdb->prefix . "php_errors WHERE error_message = '" . $formattedPlainErrorMessage . "' AND created_at >= '" . $timeToCheck . "'");

		if (count($foundErrorMessage) > 0) {
			return true;
		}
		else {
			self::removePastErrorMessages($plainErrorMessage);
			return false;
		}
	}

	private static function _addRequestDetails( $message ) {
		if ( array_key_exists( 'REQUEST_URI', $_SERVER ) ) {
			$message .= ' for ' . $_SERVER['REQUEST_URI'];
		}
		if ( array_key_exists( 'REMOTE_ADDR', $_SERVER ) ) {
			$message .= ' by ' . $_SERVER['REMOTE_ADDR'];
		}

		return $message;
	}

	private static function _getNotifyHookUrl(){
		$hookUrl = get_option(NOTIFY_PHP_ERRORS_HOOK_URL_OPTION);
		if(empty($hookUrl)){
			$hookUrl = NOTIFY_PHP_ERRORS_HOOK_URL;
		}
		return $hookUrl;
	}

}
