<?php

namespace yndenz\Plugins\NotifyYndenz;

class Message {

	/**
	 * Send a message to yndenz.
	 *
	 * @param array  $content
	 * @param bool   $fallback_to_email
	 * @param string $email_subject
	 * @param string $hook_url
	 * @param string $plainErrorMessage
	 *
	 * @return bool
	 */
	public static function send( $content, $fallback_to_email, $email_subject, $hook_url, $plainErrorMessage = '' ) {
		try {
			$initiator = debug_backtrace()[1]['class'];

			if (get_option('isNotifyUp') == 'yes') {
				return (self::_sendToNotify( $content, $hook_url, $initiator )) || ( $fallback_to_email && self::_sendToEmail( $email_subject, $content, $initiator, $plainErrorMessage ) );
			}
			else {
				return ($fallback_to_email && self::_sendToEmail( $email_subject, $content, $initiator, $plainErrorMessage ) );
			}
		} catch (\Throwable $e) {
			error_log('NotifyYndenz failed: ' . $e->getMessage());

			if ($fallback_to_email) {
				return self::_sendToEmail( $email_subject, $content, $initiator ?? 'unknown', $plainErrorMessage );
			}

			return false;
		}
	}

	/**
	 * Send the message to the yndenz notification channel.
	 *
	 * @param array $content
	 *
	 * @return bool
	 */
	private static function _sendToNotify( $content, $hook_url ) {
		$response = wp_remote_post( $hook_url, array(
			'method'    => 'POST',
			'timeout'   => 1,
			'blocking'  => true,
			'body'      => $content,
			'sslverify' => true,
		));

		if (!is_wp_error($response)) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Send the message to the yndenz notification email.
	 *
	 * @param string $email_subject
	 * @param array  $content
	 * @param string $initiator
	 * @param string $plainErrorMessage
	 *
	 * @return bool
	 */
	private static function _sendToEmail( $email_subject, $content, $initiator, $plainErrorMessage ) {
		// Check the current value of the isNotifyUp option
		if (get_option('isNotifyUp') == 'yes') {
			update_option('isNotifyUp', 'no');

			// We fire this check as the notifyLastTimeChecked option doesn't exist on the first downtime
			if (get_option('notifyLastTimeChecked') === false) {
				add_option( 'notifyLastTimeChecked', date( 'Y-m-d H:i:s' ) );
			}
		}

		// Check if the initiator is the PHPError class
		if (strpos(strtolower($initiator), 'phperror') !== false) {
			global $wpdb;

			$errorMessageFound = PHPError::_checkErrorAlreadySent($plainErrorMessage, $content['attachments'][0]['title']);

			if ($errorMessageFound) {
				return;
			}

			$errorData = [];
			$errorData['error_message'] = $plainErrorMessage;
			$errorData['created_at'] = $content['attachments'][0]['title'];

			$wpdb->insert($wpdb->prefix . 'php_errors', $errorData);
		}

		return mail( NOTIFY_EMAIL, $email_subject, print_r( $content, true ) );
	}

}
