<?php

namespace yndenz\Modules\TransientMenu;

use yndenz\Modules\BaseModule\BaseModule;

class TransientMenu extends BaseModule {

    public static function register() {
        add_filter('pre_wp_nav_menu', array(static::class, 'getMenu'), 10, 2);
        add_filter('wp_nav_menu', array(static::class, 'storeMenu'), 10, 2);
        add_action('wp_update_nav_menu', array(static::class, 'updateMenu'), 10, 1);
    }

    public static function getScripts() {
        return array(
            'nav-menu-item-script' => array('src' => get_stylesheet_directory_uri() . '/modules/transient-menu/nav-menu-item.js')
        );
    }

    /**
     * Get transient expiration time.
     *
     * @return int
     */
    public static function getCacheDuration() {
        return apply_filters('yndenz_modules_transient_menu_cache_duration', DAY_IN_SECONDS);
    }

    /**
     * Get transient expiration time.
     *
     * @return int
     */
    public static function getMenuKeyPrefix() {
        return apply_filters('yndenz_modules_transient_menu_key_prefix', 'ym-transient-menu-');
    }

    /**
     * Generate a unique id for the menu transient based on the menu arguments and currently requested page.
     *
     * @param object $args An object containing wp_nav_menu() arguments.
     *
     * @return string
     */
    public static function getMenuKey($args) {
        return static::getMenuKeyPrefix() . md5(serialize($args) . serialize(get_queried_object()));
    }

    /**
     * Get the menu transient based on menu arguments.
     *
     * @param object $args An object containing wp_nav_menu() arguments.
     *
     * @return mixed menu output if exists and valid else false.
     */
    public static function getMenuTransient($args) {
        return get_transient(static::getMenuKey($args));
    }


    /**
     * Search for the menu in the cache.
     *
     * This is a filter that lets us short-circuit the menu generation if we find it in the cache
     * so anything other than null returned will skip the menu generation.
     *
     * @param string|null $nav_menu Nav menu output to short-circuit with.
     * @param object $args An object containing wp_nav_menu() arguments.
     *
     * @return string|null
     */
    function getMenu($nav_menu, $args) {
        $in_cache = static::getMenuTransient($args);
        $last_updated = get_transient(static::getMenuKeyPrefix() . $args->theme_location . '-updated');

        if (isset($in_cache['data']) && isset($last_updated) && $last_updated < $in_cache['time']) {
            return $in_cache['data'];
        }

        return $nav_menu;
    }


    /**
     * Store menu in cache.
     *
     * @param  string $nav The HTML content for the navigation menu.
     * @param  object $args An object containing wp_nav_menu() arguments.
     *
     * @return string The HTML content for the navigation menu.
     */
    function storeMenu($nav, $args) {
        $last_updated = get_transient(static::getMenuKeyPrefix() . $args->theme_location . '-updated');
        if (!$last_updated) {
            set_transient(static::getMenuKeyPrefix() . $args->theme_location . '-updated', time());
        }

        $key = static::getMenuKey($args);
        $data = array('time' => time(), 'data' => $nav);

        set_transient($key, $data, static::getCacheDuration());

        return $nav;
    }

    /**
     * Refresh time on update to force refresh of cache.
     *
     * @param  int $menu_id
     *
     * @return void
     */
    function updateMenu($menu_id) {
        $locations = array_flip(get_nav_menu_locations());

        if (isset($locations[$menu_id])) {
            set_transient(static::getMenuKeyPrefix() . $locations[$menu_id] . '-updated', time());
        }
    }

}