<?php

namespace yndenz\Plugins\TransientWidget;

use WP_Widget;

class TransientWidget {

	public static function register() {
		add_filter( 'widget_display_callback', array( static::class, 'getWidget' ), 10, 3 );
	}

	/**
	 * Search for the menu in the cache.
	 * This is a filter that lets us short-circuit the widget generation if we find it in the cache.
	 *
	 * @param array     $instance The current widget instance's settings.
	 * @param WP_Widget $widget   The current widget instance.
	 * @param array     $args     An array of widget arguments.
	 *
	 * @return boolean
	 */
	public static function getWidget( $instance, $widget, $args ) {
		if ( false === $instance ) {
			return false;
		}

		$in_cache = static::getWidgetTransient( $instance, $args );

		if ( isset( $in_cache['data'] ) && time() - static::getCacheDuration() < $in_cache['time'] ) {
			echo $in_cache['data'];

			return false;
		}

		echo static::storeWidget( $instance, $widget, $args );

		return false;
	}

	/**
	 * Get the menu transient based on menu arguments.
	 *
	 * @param array $instance The current widget instance's settings.
	 * @param array $args     An array of widget arguments.
	 *
	 * @return mixed menu output if exists and valid else false.
	 */
	public static function getWidgetTransient( $instance, $args ) {
		return get_transient( static::getWidgetKey( $instance, $args ) );
	}

	/**
	 * Generate a unique id for the widget transient based on the widget's instance and arguments.
	 *
	 * @param array $instance The current widget instance's settings.
	 * @param array $args     An array of widget arguments.
	 *
	 * @return string
	 */
	public static function getWidgetKey( $instance, $args ) {
		$key = static::getWidgetKeyPrefix();
		$key .= md5( serialize( array( $instance, $args ) ) . serialize( get_queried_object() ) );

		return apply_filters( 'yndenz_plugins_transient_widget_key', $key, $instance, $args );
	}

	/**
	 * Get transient prefix.
	 *
	 * @return string
	 */
	public static function getWidgetKeyPrefix() {
		return apply_filters( 'yndenz_plugins_transient_widget_key_prefix', 'ym-transient-widget-' );
	}

	/**
	 * Get transient expiration time.
	 *
	 * @return int
	 */
	public static function getCacheDuration() {
		return apply_filters( 'yndenz_plugins_transient_widget_cache_duration', DAY_IN_SECONDS );
	}

	/**
	 * Store widget in cache and return it.
	 *
	 * @param array     $instance The current widget instance's settings.
	 * @param WP_Widget $widget   The current widget instance.
	 * @param array     $args     An array of widget arguments.
	 *
	 * @return string
	 */
	public static function storeWidget( $instance, $widget, $args ) {
		ob_start();
		$widget->widget( $args, $instance );
		$output = ob_get_clean();

		$key  = static::getWidgetKey( $instance, $args );
		$data = array( 'time' => time(), 'data' => $output );

		set_transient( $key, $data, static::getCacheDuration() );

		return $output;
	}

}
