<?php

abstract class CategoryConverter extends Converter {

	abstract protected function fetch_categories();

	/**
	 * Convert raw categories into WooCommerce categories
	 */
	public function convert_categories() {
		// Fetch all main categories
		$raw_categories = $this->fetch_categories();

		// Check if the result is usable
		if ($this->categories_not_usable($raw_categories)) {
			return;
		}

		$this->convert_categories_list($raw_categories);
	}

	/**
	 * Check if the raw categories are usable for conversion
	 *
	 * @param mixed $raw_categories The raw categories to be converted
	 *
	 * @return bool True if the categories are convertable, false if not
	 */
	abstract protected function categories_not_usable($raw_categories);

	/**
	 * Convert the given categories into WooCommerce product categories
	 *
	 * @param mixed $raw_categories List of categories to be converted
	 */
	abstract protected function convert_categories_list($raw_categories);

	/**
	 * Convert the given category into a WooCommerce product category
	 *
	 * @param mixed $raw_category The category to be converted
	 * @param int $parent (optional) The ID of the WooCommerce category the converted category will be a child of
	 *
	 * @return array|bool An array containing the WooCommerce product category, or false if conversion failed
	 */
	public function convert_category($raw_category, $parent = null) {
		$category_data = $this->prepare_category($raw_category, $parent);

		// If the category already exists, fetch the existing category
		$term = get_term_by('slug', $category_data['product_category']['slug'], 'product_cat');
		if ($term !== false) {
			$category = $this->api_products->get_product_category($term->term_id);
		} else {
			// Save the WooCommerce product category
			$category = $this->api_products->create_product_category($category_data);
		}

		if (is_wp_error($category)) {
			if (WP_DEBUG) {
				var_dump($category);
			}
			return false;
		}

		$this->category_add_useful_extra_data($raw_category, $category);

		$this->counter++;

		return $category;
	}

	/**
	 * Initialize the WooCommerce category
	 *
	 * @param mixed $raw_category The category to be converted
	 * @param int $parent (optional) The ID of the WooCommerce category the converted category will be a child of
	 *
	 * @return array The category data array as the WooCommerce API likes it
	 */
	abstract protected function prepare_category($raw_category, $parent = null);

	/**
	 * Add some useful extra PowerAll data WooCommerce does not support per default
	 *
	 * @param mixed $raw_category The PowerAll Webgroup to be converted
	 * @param array $new_category The converted WooCommerce product category
	 */
	abstract protected function category_add_useful_extra_data($raw_category, $new_category);

	/**
	 * Get the ID of the given category
	 *
	 * @param mixed $raw_category The category the ID be returned for
	 *
	 * @return string The ID the given category
	 */
	abstract protected function category_get_id($raw_category);

}