<?php

define('WC_API_LOCATION', WP_PLUGIN_DIR . '/woocommerce/includes/api/legacy/v3');
require_once WC_API_LOCATION . '/class-wc-api-exception.php';
require_once WC_API_LOCATION . '/class-wc-api-server.php';
require_once WC_API_LOCATION . '/class-wc-api-resource.php';
require_once WC_API_LOCATION . '/interface-wc-api-handler.php';
require_once WC_API_LOCATION . '/class-wc-api-json-handler.php';
require_once WC_API_LOCATION . '/class-wc-api-products.php';

abstract class Converter {

	protected $main_category_id;
	protected $api_products;
	protected $supported_cultures = array();
	protected $default_culture = 'nl-NL';
	protected $counter = 0;

	public function __construct() {
		// Make the WooCommerce API usable
		$server = new WC_API_Server('/wc-api/v3/');
		$this->api_products = new WC_API_Products($server);

		$this->supported_cultures = preg_split('/[\s,;]+/i', get_option('woocommerce_languages'));
		$this->default_culture = get_option('woocommerce_default_language');
	}

	public function get_counter() {
		return $this->counter;
	}

	/**
	 * Check if the total items counter is still accurate
	 * Returns just the given total items counter if the function is not overwritten
	 *
	 * @param int $total_items Total items counter
	 *
	 * @return int New total items counter
	 */
	protected function recheck_total_items($total_items) {
		return $total_items;
	}

	/**
	 * Make a plural word singular
	 *
	 * @param string $string The string to be converted
	 *
	 * @return string The converted string
	 */
	public function singular($string) {
		if (substr($string, - 1, 1) === 's') {
			$string = substr($string, 0, - 1);
		}

		return $string;
	}

    /**
     * Apply a watermark to an image
     *
     * @param resource $image The image to apply the watermark to
     * @param string $image_path Path to the file where save the image
     * @param string $watermark_path Path to the watermark image
     * @param int $margin Offset from the sides to put some space between the watermark and the image borders
     * @param string $position Position to place the watermark.
     *                         Could be either "top left", "top right", "bottom left", "bottom right", or "center".
     *                         Defaults to "center".
     */
	public function apply_watermark_to_image(&$image, $image_path, $watermark_path, $margin = 15, $position = 'center') {
        // Load the watermark
        $watermark = imagecreatefrompng($watermark_path);

        // Get the width and height of the watermark and the image
        $sizes = array(
            'image' => array(
                'width' => imagesx($image),
                'height' => imagesy($image)
            ),
            'watermark' => array(
                'width' => imagesx($watermark),
                'height' => imagesy($watermark)
            )
        );

        // Make the watermark fit the image
        $max_source_width = $sizes['image']['width'] - ($margin * 2);
        $max_source_height = $sizes['image']['height'] - ($margin * 2);
        $source_width = $sizes['watermark']['width'];
        $source_height = $sizes['watermark']['height'];
        if ($source_width > $max_source_width) {
            $source_height = ($max_source_width * $sizes['watermark']['height']) / $sizes['watermark']['width'];
            $source_width = $max_source_width;
        } else if ($source_height > $max_source_height) {
            $source_width = ($max_source_height * $sizes['watermark']['width']) / $sizes['watermark']['height'];
            $source_height = $max_source_height;
        }

        // Calculate the positioning of the watermark within the image
        $left = $margin;
        $top = $margin;
        switch ($position) {
            case 'top left':
                break;
            case 'top right':
                $left = $sizes['image']['width'] - $source_width - $margin;
                break;
            case 'bottom left':
                $top = $sizes['image']['height'] - $source_height - $margin;
                break;
            case 'bottom right':
                $left = $sizes['image']['width'] - $source_width - $margin;
                $top = $sizes['image']['height'] - $source_height - $margin;
                break;
            default: // full-sized center
                $source_width = $max_source_width;
                $source_height = ($max_source_width * $sizes['watermark']['height']) / $sizes['watermark']['width'];
                $top = ($sizes['image']['height'] - $source_height) / 2;
                break;
        }

        // Resize the watermark to fit the image
        $resized_watermark = imagecreatetruecolor($source_width, $source_height);
        imagealphablending($resized_watermark, false);
        imagesavealpha($resized_watermark, true);
        imagecopyresampled($resized_watermark, $watermark, 0, 0, 0, 0, $source_width, $source_height, $sizes['watermark']['width'], $sizes['watermark']['height']);

        // Copy the resized watermark onto the image
        imagecopy($image, $resized_watermark, $left, $top, 0, 0, $source_width, $source_height);

        // Create the directory to save the new image to if it does not exists
        $image_dir = dirname($image_path);
        if (!is_dir($image_dir)) {
            mkdir($image_dir, 0755, true);
        }

        // Save the new image
        imagejpeg($image, $image_path);

        // Free memory
        imagedestroy($watermark);
        imagedestroy($resized_watermark);
    }

}