<?php

namespace WC_Import;

abstract class ProductConverter extends Converter {

    protected $products_include_translations = false;

	public function __construct() {
		parent::__construct();
	}

	abstract protected function fetch_products($page = 1, $page_size = 10);

	abstract protected function fetch_category_products($category_id, $page = 1, $page_size = 10);

	abstract protected function fetch_product($product_id, $culture = 'nl-NL');

	/**
	 * Convert raw products related to the given category into WooCommerce products
	 * Fetches per page to minimize the delay and server load
	 *
	 * @param mixed $raw_category The category to fetch products for
	 * @param int $term_id The ID of the WooCommerce category the products will be related to
	 * @param int $page The number of the page to be fetched
	 * @param int $page_size The number of products per page to be fetched
	 *
	 * @return bool True if conversion succeeded, false if the last page is reached
	 */
	public function convert_category_products($raw_category, $term_id, $page = 1, $page_size = 10) {
		// If the category contains products, convert the products for this category
		$category_id = $this->category_get_id($raw_category);

		// Loop the conversion per page until there are no products left in this category
		return $this->convert_category_products_page($category_id, $term_id, $page, $page_size);
	}

	/**
	 * Convert a subset of PowerAll Articles related to the given category into WooCommerce products
	 *
	 * @param string $category_id The ID of the category to fetch products for
	 * @param int $term_id The ID of the WooCommerce category the products will be related to
	 * @param int $page The number of the page to be fetched
	 * @param int $page_size The number of products per page to be fetched
	 *
	 * @return bool True if conversion succeeded, false if the last page is reached
	 */
	public function convert_category_products_page($category_id, $term_id, $page, $page_size) {
		$raw_products = $this->fetch_category_products($category_id, $page, $page_size);

		// Check if the result is usable
		// If the page does not contain any products, the page number is higher than the last page
		if ($this->category_products_not_usable($raw_products)) {
			return false;
		}

		// Convert the products
		$this->convert_products_list($this->products_to_array($raw_products), $term_id);

		// If the number of products on this page is below the page size, it means we fetched the last page available
		if ($this->products_get_count($raw_products) < $page_size) {
			return false;
		}

		return true;
	}

	/**
	 * Check if the raw products are usable for conversion
	 *
	 * @param mixed $raw_products The raw products to be converted
	 *
	 * @return bool True if the products are convertable, false if not
	 */
	abstract protected function category_products_not_usable($raw_products);

	/**
	 * Convert the given products into WooCommerce products
	 *
	 * @param array $raw_products List of products to be converted
	 * @param int $term_id (optional) The ID of the WooCommerce category the products will be related to
	 */
	protected function convert_products_list($raw_products, $term_id = null) {
		foreach ($raw_products as $raw_product) {
			if ($this->product_is_convertable($raw_product)) {
				$default_culture_product_id = $this->convert_product($raw_product, $term_id);

				if ($default_culture_product_id === false || is_wp_error($default_culture_product_id)) {
					continue;
				}

				foreach ($this->supported_cultures as $culture) {
					// Skip the default culture, it's already converted above
					if ($culture === $this->default_culture) {
						continue;
					}

					if ($this->products_include_translations) {
					    $result = $raw_product;
                    } else {
                        $product_id = $this->product_get_id($raw_product);
                        $translated_raw_product = $this->fetch_product($product_id, $culture);

                        // Check if the result is usable
                        $result = $this->product_get_result($translated_raw_product);
                        if ($result === false) {
                            continue;
                        }
                    }

					// Convert the product
					$lang = substr($culture, 0, 2);
					$this->convert_product($result, $term_id, $lang, $default_culture_product_id);
				}
			}
		}
	}

	/**
	 * Check if the raw product is convertable
	 *
	 * @param mixed $raw_product The product to be converted
	 *
	 * @return bool True if the raw product is convertable, false if not
	 */
	abstract protected function product_is_convertable($raw_product);

	/**
	 * Check if the raw product is usable for conversion and return the result
	 *
	 * @param mixed $raw_product The product to be converted
	 *
	 * @return mixed The raw product if the raw product is convertable, false if not
	 */
	abstract protected function product_get_result($raw_product);

	/**
	 * Get the unique identifier of the raw product
	 *
	 * @param mixed $raw_product The product the unique identifier should be returned for
	 *
	 * @return string The unique identifier of the raw product
	 */
	abstract protected function product_get_id($raw_product);

	/**
	 * Get the unique identifier of the given category
	 *
	 * @param mixed $raw_category The category the unique identifier be returned for
	 *
	 * @return string The unique identifier of the raw category
	 */
	abstract protected function category_get_id($raw_category);

	/**
	 * Get the number of products in the given category
	 *
	 * @param mixed $raw_category The category the ID be returned for
	 *
	 * @return string The number of products in the given category
	 */
	abstract protected function category_get_product_count($raw_category);

	/**
	 * Convert PowerAll Articles into WooCommerce products
	 * Fetches per page to minimize the delay and server load
	 *
	 * @param int $page The number of the page to be fetched
	 * @param int $page_size The number of products per page to be fetched
	 *
	 * @return bool True if conversion succeeded, false if the feed is not usable or the last page is reached
	 */
	public function convert_products($page, $page_size) {
		$raw_products = $this->fetch_products($page, $page_size);

		// Check if the result is usable
		// If the page does not contain any products, the page number is higher than the last page
		if ($this->products_not_usable($raw_products)) {
			return false;
		}

		// Convert the products
		$this->convert_products_list($this->products_to_array($raw_products));

		if ($this->products_get_count($raw_products) < $page_size) {
			return false;
		}

		// Check if we fetched the last page available
		$total_items = $this->recheck_total_items(PHP_INT_MAX);
		if ($page * $page_size >= $total_items) {
			return false;
		}

		return true;
	}

	/**
	 * Convert a single PowerAll Article into a WooCommerce product
	 *
	 * @param string $product_id The ID of the PowerAll Article to be converted
	 *
	 * @return mixed ID of the WooCommerce product created, false if the product was not found, or an error if conversion failed
	 */
	public function convert_single_product($product_id) {
		$raw_product = $this->fetch_product($product_id);

		// Check if the result is usable
		$result = $this->product_get_result($raw_product);
		if ($result === false) {
			return false;
		}

		// Convert the product
		return $this->convert_product($result);
	}

	/**
	 * Check if the raw products are usable for conversion
	 *
	 * @param mixed $raw_products The raw products to be converted
	 *
	 * @return bool True if the products are convertable, false if not
	 */
	abstract protected function products_not_usable($raw_products);

	/**
	 * Get the list of products as an array from the raw request data
	 * Just returns the raw request data itself per default, assuming it's an array already
	 *
	 * @param mixed $raw_products The raw products to be converted
	 *
	 * @return array The list of products as an array
	 */
	protected function products_to_array($raw_products) {
		if (is_array($raw_products)) {
			return $raw_products;
		} else {
			return array($raw_products);
		}
	}

	/**
	 * Get the number of products in the raw products list
	 *
	 * @param mixed $raw_products List of raw products to be converted
	 *
	 * @return int The number of products
	 */
	abstract protected function products_get_count($raw_products);

	/**
	 * Convert a single PowerAll Article into a WooCommerce product
	 *
	 * @param mixed $raw_product The PowerAll Article to be converted
	 * @param int $term_id (optional) The ID of the WooCommerce category the product will be related to
	 * @param string $lang The language of the product
	 * @param null $translation_of (optional) The ID of the WooCommerce product the product is a translation of
	 *
	 * @return mixed ID of the WooCommerce product created, or an error if conversion failed
	 */
	public function convert_product($raw_product, $term_id = null, $lang = 'nl', $translation_of = null) {
		$product_data = $this->prepare_product($raw_product, $term_id, $lang, $translation_of);

		$this->product_add_categories($raw_product, $product_data);
		$this->product_add_images($raw_product, $product_data);

		// Save the WooCommerce product
		// If the product already existed, edit the product data to make sure the product is up-to-date
		$product_id = wc_get_product_id_by_sku($product_data['product']['sku']);
		if ($product_id) {
			$new_product = $this->api_products->edit_product($product_id, $product_data);
		} else {
			$new_product = $this->api_products->create_product($product_data);
		}

		if (is_wp_error($new_product)) {
            if (WP_DEBUG) {
                var_dump($new_product);
            }
			return $new_product;
		}

		$this->prevent_image_duplication($product_data, $new_product);
		$this->product_add_downloads($raw_product, $new_product);
		$this->product_add_useful_extra_data($raw_product, $new_product);

		$this->counter ++;

		return $new_product['product']['id'];
	}

	/**
	 * Initialize the WooCommerce product data array
	 *
	 * @param mixed $raw_product The raw product to be converted
	 * @param int $term_id (optional) The ID of the WooCommerce category the product will be related to
	 * @param string $lang The language of the product
	 * @param null $translation_of (optional) The ID of the WooCommerce product the product is a translation of
	 *
	 * @return array The product data array as the WooCommerce API likes it
	 */
	abstract protected function prepare_product($raw_product, $term_id = null, $lang = 'nl', $translation_of = null);

	/**
	 * If the product is related to a category, search for the converted WooCommerce category and connect them
	 *
	 * @param mixed $raw_product The raw product to be converted
	 * @param array $product_data The conversion product data to be extended
	 */
	abstract protected function product_add_categories($raw_product, &$product_data);

	/**
	 * If the product contains images, map them to the product
	 * If the image already exists, found by PowerAll image URL, use the original image to prevent image duplication
	 *
	 * @param mixed $raw_product The raw product to be converted
	 * @param array $product_data The conversion product data to be extended
	 */
	abstract protected function product_add_images($raw_product, &$product_data);

	/**
	 * Prevent product image duplication by saving the PowerAll image URL of the images
	 * Next time the conversion is running we can find the original product image by the PowerAll image URL
	 *
	 * @param array $product_data The conversion product data
	 * @param array $new_product The converted WooCommerce product
	 */
	protected function prevent_image_duplication($product_data, $new_product) {
		foreach ($new_product['product']['images'] as $index => $image) {
			if (!empty($image['id']) && $image['id'] > 0 && array_key_exists($index, $product_data['product']['images'])) {
				$image_source = $product_data['product']['images'][$index];
				update_post_meta($image['id'], 'original_image_url', $image_source['src']);
			}
		}
	}

	/**
	 * Add downloadable files to the WooCommerce product
	 *
	 * @param mixed $raw_product The raw product to be converted
	 * @param array $new_product The converted WooCommerce product
	 */
	abstract protected function product_add_downloads($raw_product, $new_product);

	/**
	 * Add some useful extra PowerAll data WooCommerce does not support per default
	 *
	 * @param mixed $raw_product The raw product to be converted
	 * @param array $new_product The converted WooCommerce product
	 */
	abstract protected function product_add_useful_extra_data($raw_product, $new_product);

    /**
     * Set the primary category for the product, which could be used to build the perfect SEO URL for the product
     *
     * @param array $product The converted WooCommerce product
     */
	protected function product_set_primary_category($product) {
        // Check if the primary category is already set
        // This means the product is being edited, not newly created
        $primary_term = get_post_meta($product['product']['id'], '_yoast_wpseo_primary_product_cat', true);
        if (!empty($primary_term)) {
            return;
        }

        $product_categories = wp_get_post_terms($product['product']['id'], 'product_cat', array('fields' => 'ids'));

        // If there's more than just the main category, find a more relevant category
        if (count($product_categories) > 1) {
            $product_categories = array_filter($product_categories, function ($term_id) {
                return $term_id !== $this->main_category_id;
            });
        }

        if (count($product_categories) > 0) {
            // Pick the first listed category to become the primary category
            update_post_meta($product['product']['id'], '_yoast_wpseo_primary_product_cat', current($product_categories));
        }
    }

}