<?php

namespace WC_Import;

use Exception;

class WatermarkImage {

	protected $file, $destination, $destination_info, $sizes, $resource, $watermark, $margin = 15;

	/**
	 * Create a new job instance.
	 *
	 * @param string $file           The path to the image to be processed
	 * @param string $destination    The path to where the image should be stored
	 * @param string $watermark_path Path the watermark image file
	 *
	 * @throws Exception
	 */
	public function __construct( $file, $destination, $watermark_path ) {
		$this->file        = $file;
		$this->destination = $destination;

		$this->destination_info = pathinfo( $this->destination );

		if ( ! is_dir( $this->destination_info['dirname'] ) ) {
			mkdir( $this->destination_info['dirname'], 0755, true );
		}

		$this->save_copy();
		$this->apply_watermark( $watermark_path );
		$this->clear_memory();
	}

	/**
	 * Save a copy of the resource image in the destination
	 */
	protected function save_copy() {
		$data = file_get_contents( $this->file );
		if ( $data === false ) {
			throw new Exception( 'failed to fetch image' );
		}

		if ( empty( $data ) ) {
			throw new Exception( 'image file is empty' );
		}

		$this->resource = imagecreatefromstring( $data );
		if ( $this->resource === false ) {
			throw new Exception( 'failed to create resource' );
		}

		if ( $this->image_file_type_from_binary( $data ) === 'image/png' ) {
			$input          = $this->resource;
			$width          = imagesx( $input );
			$height         = imagesy( $input );
			$this->resource = imagecreatetruecolor( $width, $height );
			$white          = imagecolorallocate( $this->resource, 255, 255, 255 );
			imagefilledrectangle( $this->resource, 0, 0, $width, $height, $white );
			imagecopy( $this->resource, $input, 0, 0, 0, 0, $width, $height );
		}

		// Save the image
		imagejpeg( $this->resource, $this->destination );
	}

	protected function image_file_type_from_binary( $binary ) {
		if ( ! preg_match(
			'/\A(?:(\xff\xd8\xff)|(GIF8[79]a)|(\x89PNG\x0d\x0a)|(BM)|(\x49\x49(?:\x2a\x00|\x00\x4a))|(FORM.{4}ILBM))/',
			$binary, $hits
		) ) {
			return 'application/octet-stream';
		}
		static $type = array(
			1 => 'image/jpeg',
			2 => 'image/gif',
			3 => 'image/png',
			4 => 'image/x-windows-bmp',
			5 => 'image/tiff',
			6 => 'image/x-ilbm',
		);

		return $type[ count( $hits ) - 1 ];
	}

	/**
	 * Apply a watermark to the image
	 *
	 * @param string $watermark_path Path the watermark image file
	 *
	 * @throws Exception
	 */
	protected function apply_watermark( $watermark_path ) {
		// Load the watermark
		if ( ! file_exists( $watermark_path ) ) {
			throw new Exception( 'watermark image does not exist' );
		}

		if ( ! is_dir( $this->destination_info['dirname'] . '/watermark' ) ) {
			mkdir( $this->destination_info['dirname'] . '/watermark', 0755, true );
		}

		$this->watermark = imagecreatefrompng( $watermark_path );

		$this->measure_images();
		$this->resize_watermark();

		// Calculate the positioning of the watermark within the image
		if ( $this->sizes['image']['ratio'] < $this->sizes['watermark']['ratio'] ) {
			$left = $this->margin;
			$top  = ( $this->sizes['image']['height'] - $this->sizes['watermark']['height'] ) / 2;
		} else {
			$left = ( $this->sizes['image']['width'] - $this->sizes['watermark']['width'] ) / 2;
			$top  = $this->margin;
		}

		// Copy the resized watermark onto the image
		imagecopy( $this->resource, $this->watermark, $left, $top, 0, 0, $this->sizes['watermark']['width'], $this->sizes['watermark']['height'] );

		// Save the new image
		imagejpeg( $this->resource, $this->destination_info['dirname'] . '/watermark/' . $this->destination_info['basename'] );
	}

	/**
	 * Get the width and height of the watermark and the image
	 */
	protected function measure_images() {
		$this->sizes                       = array(
			'image'     => array(
				'width'  => imagesx( $this->resource ),
				'height' => imagesy( $this->resource ),
			),
			'watermark' => array(
				'width'  => imagesx( $this->watermark ),
				'height' => imagesy( $this->watermark )
			)
		);
		$this->sizes['image']['ratio']     = $this->sizes['image']['width'] / $this->sizes['image']['height'];
		$this->sizes['watermark']['ratio'] = $this->sizes['watermark']['width'] / $this->sizes['watermark']['height'];
	}

	/**
	 * Resize the watermark to fit the image
	 */
	protected function resize_watermark() {
		$max_source_width  = $this->sizes['image']['width'] - ( $this->margin * 2 );
		$max_source_height = $this->sizes['image']['height'] - ( $this->margin * 2 );
		$source_width      = $this->sizes['watermark']['width'];
		$source_height     = $this->sizes['watermark']['height'];

		// calculate the new size of the watermark
		if ( $this->sizes['image']['ratio'] < $this->sizes['watermark']['ratio'] ) {
			$this->sizes['watermark']['height'] = ( $max_source_width * $source_height ) / $source_width;
			$this->sizes['watermark']['width']  = $max_source_width;
		} else {
			$this->sizes['watermark']['width']  = ( $max_source_height * $source_width ) / $source_height;
			$this->sizes['watermark']['height'] = $max_source_height;
		}

		$resized_watermark = imagecreatetruecolor( $this->sizes['watermark']['width'], $this->sizes['watermark']['height'] );
		imagealphablending( $resized_watermark, false );
		imagesavealpha( $resized_watermark, true );
		imagecopyresampled( $resized_watermark, $this->watermark, 0, 0, 0, 0, $this->sizes['watermark']['width'], $this->sizes['watermark']['height'], $source_width, $source_height );

		// overwrite the watermark resource with the resized version
		imagedestroy( $this->watermark );
		$this->watermark = $resized_watermark;
	}

	/**
	 * Free some memory to make sure this image won't cause memory leaks or memory exhaustion
	 */
	protected function clear_memory() {
		if ( is_resource( $this->watermark ) ) {
			imagedestroy( $this->watermark );
		}
		if ( is_resource( $this->resource ) ) {
			imagedestroy( $this->resource );
		}
	}

}
