<?php
/*
Plugin Name: XML WP_Query
Plugin URI:  http://www.yndenz.com
Description: Query an XML feed the same way as using WP_Query
Version:     2.2.1
Author:      Gido Manders <g.manders@yndenz.com>
Author URI:  http://www.yndenz.com
Domain Path: /languages
Text Domain: xml-wp-query
*/

defined('ABSPATH') or die('No script kiddies please!');

if (!defined('XML_WP_QUERY_FEED_URL_PROBLEM')) {
    define('XML_WP_QUERY_NO_FEED_URL', 404);
    define('XML_WP_QUERY_CACHE_PROBLEM', 409);
    define('XML_WP_QUERY_READ_XML_PROBLEM', 410);
    define('XML_WP_QUERY_FEED_URL_PROBLEM', 503);
}

require_once 'settings.php';

/**
 * Load the text domain for the XML WP_Query plugin
 */
add_action('plugins_loaded', 'xml_wp_query_load_textdomain');
function xml_wp_query_load_textdomain()
{
    load_plugin_textdomain('xml-wp-query', FALSE, basename(dirname(__FILE__)) . '/languages/');
}

/**
 * Force update the cache
 * Used by the button "update now" in the XML WP_Query Settings page
 */
add_action('wp_ajax_xml_wp_query_update_cache', 'xml_wp_query_update_cache');
add_action('wp_ajax_nopriv_xml_wp_query_update_cache', 'xml_wp_query_update_cache');
function xml_wp_query_update_cache()
{
    new XML_WP_Query($_POST['id'], '', true);
    echo get_post_meta($_POST['id'], 'xml_wp_query_cache_last_update', true);
    exit;
}

function xml_wp_query_random_string($length = 10)
{
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charactersLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, $charactersLength - 1)];
    }
    return $randomString;
}

function xml_wp_query_compare($a, $b, $order) {
    $value_a = $a;
    $value_b = $b;
    if (is_string($value_a) && is_string($value_b)) {
        $value_a = iconv('UTF-8', 'ASCII//TRANSLIT', strval($value_a));
        $value_b = iconv('UTF-8', 'ASCII//TRANSLIT', strval($value_b));
    }
    if ($value_a == $value_b) {
        return 0;
    }
    if ($order === 'desc') {
        return ($value_a > $value_b) ? -1 : 1;
    } else {
        return ($value_a < $value_b) ? -1 : 1;
    }
}

/**
 * Returns an array representation of a DOMNode
 *
 * @param DOMDocument $dom
 * @param DOMNode $node
 * @return array
 */
function node_to_array($dom, $node)
{
    // Return false if $dom is not a DOM element
    if (!is_a($dom, 'DOMDocument') || !is_a($node, 'DOMNode')) {
        return false;
    }
    $array = [];

    // Discard empty nodes
    $localName = trim($node->localName);
    if (empty($localName)) {
        return false;
    }

    if (XML_TEXT_NODE == $node->nodeType) {
        return [$node->nodeValue];
    }

    foreach ($node->attributes as $attr) {
        $array['@' . $attr->localName] = $attr->nodeValue;
    }

    foreach ($node->childNodes as $childNode) {
        if (1 == $childNode->childNodes->length && XML_TEXT_NODE == $childNode->firstChild->nodeType) {
            $array[$childNode->localName] = $childNode->nodeValue;
        } else {
            if (false !== ($a = nodeToArray($dom, $childNode))) {
                $array[$childNode->localName] = $a;
            }
        }
    }
    return $array;
}

class XML_WP_Query extends WP_Query
{

    public $feed;
    public $xml_nodes = array();
    protected $cache_filename;

    /**
     * Constructor.
     *
     * Sets up the XML_WP_Query query, if parameter query is not empty.
     * Fetches XML feed and writes it to cache file, if parameter force_cache_update is false.
     * Define XML_WP_QUERY_CACHE_DIR to change the cache file directory, defaults to uploads_dir.
     * Define XML_WP_QUERY_CACHE_FILE and set $cache_file_only true to always use the XML_WP_QUERY_CACHE_FILE.
     *
     * @since 0.1.0
     * @access public
     *
     * @param string $id ID of the XML feed.
     * @param string|array $query URL query string or array of vars.
     * @param bool $force_cache_update Force an update of the feed cache file if caching is enabled.
     * @param bool $cache_file_only Force using the cache file only. Useful when there's no external feed URL.
     *
     * @throws Exception
     */
    public function __construct($id, $query = '', $force_cache_update = false, $cache_file_only = false)
    {
        $this->setup($id, $query, $force_cache_update, $cache_file_only);
    }

    /**
     * Set up the XML_WP_Query query.
     * Fetches XML feed and writes it to cache file, if parameter force_cache_update is false.
     * Define XML_WP_QUERY_CACHE_DIR to change the cache file directory, defaults to uploads_dir.
     * Define XML_WP_QUERY_CACHE_FILE and set $cache_file_only true to always use the XML_WP_QUERY_CACHE_FILE.
     *
     * @param int $id ID of the XML feed.
     * @param string|array $query URL query string or array of vars.
     * @param bool $force_cache_update Force an update of the feed cache file if caching is enabled.
     * @param bool $cache_file_only Force using the cache file only. Useful when there's no external feed URL.
     *
     * @throws Exception
     */
    public function setup($id, $query, $force_cache_update = false, $cache_file_only = false)
    {
        // Store feed url
        $this->feed = get_post($id);
        $this->xml_nodes = array();

        // Set the XML by fetching the feed
        $this->getXml($force_cache_update, $cache_file_only);

        $this->query($query);
    }

    /**
     * Get the XML.
     * Uses the cached file if it exists and is valid.
     * Fetches XML feed and writes it to the cache file if necessary, or if parameter force_cache_update is true.
     *
     * Define XML_WP_QUERY_CACHE_DIR to change the cache file directory, defaults to uploads_dir.
     * Define XML_WP_QUERY_CACHE_FILE and set $cache_file_only true to always use the XML_WP_QUERY_CACHE_FILE.
     *
     * @param bool $force_cache_update Force an update of the feed cache file if caching is enabled.
     * @param bool $cache_file_only Force using the cache file only.
     *
     * @throws Exception An exception is thrown if the cache could not be updated or the XML is invalid.
     */
    public function getXml($force_cache_update = false, $cache_file_only = false)
    {
        $feed_url = get_post_meta($this->feed->ID, 'xml_wp_query_feed_url', true);
        $use_cache = get_post_meta($this->feed->ID, 'xml_wp_query_cache', true);

        if ($cache_file_only) {
            // if the cache file is converted successfully, we don't have to continue
            if ($this->read_cache()) {
                return;
            }
        }

        // if cache is force updated or cache is invalid/outdated, update cache
        if ($use_cache && !$force_cache_update && $this->is_cache_valid()) {
            if ($this->read_cache()) {
                return;
            }
        }

        // if feed url is empty, use the cache file
        if (empty($feed_url)) {
            $feed_url = $this->get_cache_filename();
        }

        $feed_url = $this->fetch_feed($feed_url);

        // if the fetch function returns false, the cache feed is used, so we don't have to continue
        if ($feed_url === false) {
            return;
        }

        try {
            $changes_applied = $this->readXML($feed_url, true);
        } catch (Exception $e) {

            $message = 'De cache van de XML feed van ' . site_url() . ' is invalid. Het is geen dringend probleem, de backup wordt gebruikt, maar kijk er even naar.';
            wp_mail('developers@yndenz.com', get_bloginfo('name') . ' - XML feed cache probleem', $message);
        }

        // update the cache file if necessary
        if ($use_cache && ($force_cache_update || $changes_applied)) {
            try {
                $this->update_cache();
            } catch (Exception $e) {
                if ($e->getCode() !== XML_WP_QUERY_FEED_URL_PROBLEM) {
                    $e = new Exception(_('XML WP_Query - Could not write feed to cache', 'xml-wp-query'), XML_WP_QUERY_CACHE_PROBLEM);
                }
                throw $e;
            }
        }
    }

    /**
     * Check if the cache file exists and is not out-of-date.
     *
     * @return bool
     */
    public function is_cache_valid() {
        $cache_filename = $this->get_cache_filename() . '.cache';

        // if cache file does not exists or it does not contain at least a few lines, it's not valid to use
        if (!file_exists($cache_filename) || filesize($cache_filename) < 100) {
            return false;
        }

        $last_cache_update = (int)get_post_meta($this->feed->ID, 'xml_wp_query_cache_last_update', true);
        $cache_interval = get_post_meta($this->feed->ID, 'xml_wp_query_cache_interval', true);

        // if the cache file is not up-to-date, don't use it
        if ($last_cache_update < strtotime('-' . $cache_interval)) {
            return false;
        }

        return true;
    }

    /**
     * Get the cache filename.
     * If it's an external feed, this is a locally cached version of the feed.
     * If it's not an external feed, this is actually the original feed.
     *
     * @return string
     */
    public function get_cache_filename() {
        if (!empty($this->cache_filename)) {
            return $this->cache_filename;
        }

        if (defined('XML_WP_QUERY_CACHE_FILE')) {
            $this->cache_filename = XML_WP_QUERY_CACHE_FILE;
        } else {
            if (!defined('XML_WP_QUERY_CACHE_DIR')) {
                define('XML_WP_QUERY_CACHE_DIR', defined('UPLOADS') ? UPLOADS : WP_CONTENT_DIR . '/uploads');
            }
            $this->cache_filename = XML_WP_QUERY_CACHE_DIR . '/' . $this->feed->post_name . '.xml';
        }

        return $this->cache_filename;
    }

    /**
     * Fetch the feed.
     * IF the feed is from an external source, we save a local copy.
     * If the feed could not be fetched or the response is empty, we try to use the cache.
     * If the cache doesn't work, we fall back to the backup file (previously fetched feed).
     *
     * @param $feed_url
     *
     * @return bool|string Returns false if the cache is used, otherwise returns the location of the local copy.
     * @throws Exception Throws an exception if really no useable XML could be found.
     */
    public function fetch_feed($feed_url) {
        try {
            $original_content = file_get_contents($feed_url);

            if (empty($original_content)) {
                throw new Exception(__('XML WP_Query - Could not resolve feed URL', 'xml-wp-query'), XML_WP_QUERY_FEED_URL_PROBLEM);
            }

            // If the feed is from an external source, save a local copy
            $cache_file = $this->get_cache_filename();
            if ($feed_url !== $cache_file) {
                if (!file_put_contents($feed_url, $original_content)) {
                    throw new Exception(__('XML WP_Query - Could not resolve feed URL', 'xml-wp-query'), XML_WP_QUERY_FEED_URL_PROBLEM);
                }
            }

            // we don't want to touch the fetched feed, so we use a copy
            copy($feed_url, $feed_url . '.orig');
            $feed_url = $feed_url . '.orig';
        } catch (Exception $e) {
            $message = "De XML feed van " . site_url() . " kan niet geüpdatet worden.\r\n";
            $subject = get_bloginfo('name') . ' - ';

            // try to use the cache file
            if ($this->read_cache()) {
                $message .= 'Het is geen dringend probleem, de cache wordt gebruikt, maar kijk er even naar.';
                wp_mail('developers@yndenz.com', $subject . 'XML feed probleem', $message);
                return false;
            }

            // if a backup exists, use the backup
            if (!file_exists($this->get_cache_filename() . '.orig')) {
                // we really cannot find any useable XML
                $message = 'De cache kan ook niet gebruikt worden, dus er moet per direct actie ondernomen worden!';
                wp_mail('developers@yndenz.com', $subject . 'Dringend XML feed probleem!', $message);

                throw new Exception(__('XML WP_Query - Could not resolve feed URL', 'xml-wp-query'), XML_WP_QUERY_FEED_URL_PROBLEM);
            }

            $message = 'Het is geen dringend probleem, een backup wordt gebruikt, maar kijk er even naar.';
            wp_mail('developers@yndenz.com', $subject . 'XML feed probleem', $message);
            $feed_url = $this->get_cache_filename() . '.orig';
        }

        return $feed_url;
    }

    /**
     * Try to read the cached XML.
     *
     * @return bool Returns true if the cache file is valid, false otherwise.
     */
    public function read_cache() {
        $cache = $this->get_cache_filename() . '.cache';
        if (!file_exists($cache)) {
            return false;
        }
        try {
            $this->readXML($cache);
        } catch (Exception $e) {
            $backup = $this->get_cache_filename() . '.bak';
            if (!file_exists($backup)) {
                return false;
            }

            try {
                $this->readXML($backup);

                // overwrite the cache with the backup to prevent spamming about using the backup
                copy($backup, $cache);

                $message = 'De cache van de XML feed van ' . site_url() . ' is invalid. Het is geen dringend probleem, de backup wordt gebruikt, maar kijk er even naar.';
                wp_mail('developers@yndenz.com', get_bloginfo('name') . ' - XML feed cache probleem', $message);
            } catch (Exception $e) {
                return false;
            }
        }

        return true;
    }

    /**
     * Try to read the XML in the given file.
     *
     * @param $filename
     *
     * @return bool Returns true if changes were made to the XML, false otherwise.
     * @throws Exception Throws an exception if no useable XML tags could be found.
     */
    public function readXML($filename, $cleanup = false)
    {
        $changes_applied = false;
        $this->xml_nodes = array();
        $itemTag = get_post_meta($this->feed->ID, 'xml_wp_query_item_tag', true);

        $z = new XMLReader;
        $z->open($filename);

        // skip everything until we find the first item
        while ($z->read() && $z->name !== $itemTag) ;

        while ($z->name === $itemTag && $z->nodeType === XMLReader::ELEMENT) {
            $item = new SimpleXMLElement($z->readOuterXML());

            if ($cleanup) {
                $cleaned_up = $this->cleanup_xml($item);
                $tags_added = $this->add_identifier_tags($item);

                $changes_applied = $changes_applied || $cleaned_up || $tags_added;
            }

            array_push($this->xml_nodes, $item);

            // hop to the next item until the end of the tree
            $z->next($itemTag);
        }

        // if $xml_data is still empty, no XML feed was found
        if (empty($this->xml_nodes)) {
            throw new Exception(__('XML WP_Query - XML is empty', 'xml-wp-query'), XML_WP_QUERY_READ_XML_PROBLEM);
        }

        return $changes_applied;
    }

    /**
     * Clean up the XML to speed up queries.
     *
     * @param $item
     * @return bool
     */
    public function cleanup_xml(&$item)
    {
        $cleanup_query = get_post_meta($this->feed->ID, 'xml_wp_query_cleanup_query', true);

        // If no cleanup query is set, return false
        if (empty($cleanup_query)) {
            return false;
        }

        // If no items were found, return false
        if (empty($item)) {
            return false;
        }

        // Removed any unnecessary tags
        $changes_applied = false;
        $cleanup_tags = $item->xpath($cleanup_query);
        if (count($cleanup_tags)) {
            $changes_applied = true;

            foreach ($cleanup_tags as $cleanup_tag) {
                unset($cleanup_tag[0]);
            }
        }

        return $changes_applied;
    }

    /**
     * Add the date, ID and slug the the specified item to speed up queries.
     *
     * @param $item
     * @return bool
     */
    public function add_identifier_tags(&$item)
    {
        $slug = $item->xpath('//slug');
        $post_date = $item->xpath('//post_date');
        $ID = $item->xpath('//ID');

        // If the XML already contains the post_date, the ID and add a slug, don't update it
        if (!empty($slug) && !empty($post_date) && !empty($ID)) {
            return false;
        }

        $changes_applied = false;

        if (empty($ID)) {
            $changes_applied = true;
            $this->add_id($item);
        }

        if (empty($post_date)) {
            $changes_applied = true;
            $this->add_date($item);
        }

        if (empty($slug)) {
            $changes_applied = true;
            $this->add_slug($item);
        }

        return $changes_applied;
    }

    /**
     * Add ID to the specified item.
     *
     * @param $item
     */
    private function add_id(&$item) {
        $uniqueKeyTag = get_post_meta($this->feed->ID, 'xml_wp_query_item_unique_key_tag', true);
        if ($uniqueKeyTag && property_exists($item, $uniqueKeyTag)) {
            $ID = strval($item->$uniqueKeyTag);
        } else if ($uniqueKeyTag) {
            $ID = $item->xpath($uniqueKeyTag . '/text()');
        } else if (property_exists($item, 'id')) {
            $ID = strval($item->id);
        } else if (property_exists($item, 'ID')) {
            $ID = strval($item->ID);
        } else {
            $ID = rand(10000, 100000000);
        }
        $item->ID = $ID;
    }

    /**
     * Add date to the specified item.
     *
     * @param $item
     */
    private function add_date(&$item) {
        $dateTag = get_post_meta($this->feed->ID, 'xml_wp_query_item_date_tag', true);
        $date = null;
        if ($dateTag && property_exists($item, $dateTag)) {
            $date = date('Y-m-d H:i:s', strtotime(strval($item->$dateTag)));
        } else if ($dateTag) {
            $date = date('Y-m-d H:i:s', strtotime($item->xpath($dateTag . '/text()')));
        } else if (property_exists($item, 'date')) {
            $date = date('Y-m-d H:i:s', strtotime(strval($item->date)));
        } else if (property_exists($item, 'created_at')) {
            $date = date('Y-m-d H:i:s', strtotime(strval($item->created_at)));
        } else if (property_exists($item, 'updated_at')) {
            $date = date('Y-m-d H:i:s', strtotime(strval($item->updated_at)));
        }
        $item->post_date = $date;
    }

    /**
     * Add slug to the specified item.
     *
     * @param $item
     */
    private function add_slug(&$item) {
        // Define the tags used to generate the slug
        $slugTags = explode(',', get_post_meta($this->feed->ID, 'xml_wp_query_item_slug_tags', true));

        // Loop the tags to get the data
        $slugs = [];
        foreach ($slugTags as $slugTag) {
            if (property_exists($item, $slugTag)) {
                array_push($slugs, strval($item->$slugTag));
            } else {
                $tag = $item->xpath($slugTag . '/text()');
                if (is_array($tag)) {
                    $tag = $tag[0];
                }
                $slugs[] = strval($tag);
            }
        }

        // Apply filters if any was registered (filters i.e. limit the length of the slug)
        $slugs = apply_filters('xml_wp_query_slug', $slugs);
        $slug = sanitize_title(implode(' ', $slugs));
        $check_slug = $slug;

        // If there are any items with a similar slug, append a postfix
        $i = 1;
        while ($this->check_slug($check_slug)) {
            if ($i === 1 && property_exists($item, 'ID') && !empty($item->ID)) {
                $slug .= '-' . $item->ID;
                $check_slug = $slug;
            } else {
                $check_slug = $slug . '-' . $i;
            }
            $i++;
        }

        $item->slug = $check_slug;
    }

    /**
     * Check if the slug already exists.
     *
     * @param $slug
     * @return bool
     */
    private function check_slug($slug) {
        $similar_slug_found = false;

        foreach ($this->xml_nodes as $node) {
            if (strval($node->slug) === $slug) {
                $similar_slug_found = true;
                break;
            }
        }

        return $similar_slug_found;
    }

    /**
     * Save the XML to the cache file and make a backup.
     *
     * @throws Exception Throws an exception if an empty node was found, causing the XML to become invalid.
     */
    public function update_cache() {
        $tmp_filename = $this->get_cache_filename() . '.tmp';

        // create an empty temporary cache file
        file_put_contents($tmp_filename, '<?xml version="1.0" encoding="UTF-8"?><feed>');

        foreach ($this->xml_nodes as $item) {
            $xml = $item->asXML();
            $xml = str_replace('<?xml version="1.0"?>', '', $xml);
            $xml = preg_replace('~\s*(<([^-->]*)>[^<]*<!--\2-->|<[^>]*>)\s*~', '$1', $xml);

            // if new cache string is empty, throw an error
            if (empty($xml)) {
                throw new Exception(_('XML WP_Query - Feed contains invalid XML', 'xml-wp-query'), XML_WP_QUERY_FEED_URL_PROBLEM);
            }

            // write the item to the cache
            file_put_contents($tmp_filename, $xml, FILE_APPEND);
        }

        file_put_contents($tmp_filename, '</feed>', FILE_APPEND);

        // make the temporary file the permanent new feed and save a backup
        copy($tmp_filename, $this->get_cache_filename() . '.cache');
        copy($tmp_filename, $this->get_cache_filename() . '.bak');

        update_post_meta($this->feed->ID, 'xml_wp_query_cache_last_update', time());
    }

    /**
     * Retrieve the posts based on query variables.
     * There are a few filters and actions that can be used to modify the post
     * database query.
     *
     * @return array List of posts
     * @throws Exception If item tag is empty,
     */
    public function get_posts()
    {
        $items = array_filter($this->xml_nodes, function ($item) {
            $item_meets_base_requirements = $this->compare_id($item) && $this->compare_date($item) && $this->compare_slug($item);
            if ($item_meets_base_requirements && (array_key_exists('meta_query', $this->query_vars) || array_key_exists('meta_key', $this->query_vars) || array_key_exists('key', $this->query_vars))) {
                $item_meets_base_requirements = $this->compare_meta($item);
            }
            return $item_meets_base_requirements;
        });

        // If no items were found, return an empty array
        if (empty($items)) {
            return array();
        }

        // Loop the items
        $posts = [];
        foreach ($items as $item) {
            // Fetch the title
            $titleTag = get_post_meta($this->feed->ID, 'xml_wp_query_item_title_tag', true);
            if ($titleTag && property_exists($item, $titleTag)) {
                $item->post_title = strval($item->$titleTag);
            } else if ($titleTag) {
                $item->post_title = $item->xpath($titleTag . '/text()');
            } else if (property_exists($item, 'title')) {
                $item->post_title = strval($item->title);
            } else if (property_exists($item, 'slug')) {
                $item->post_title = strval($item->slug);
            }
            
            if (array_key_exists('fields', $this->query_vars)) {
                $this->pluck($posts, $item);
                continue;
            }

            // Convert item into a WP_Post object
            $post = new WP_Post(new stdClass());
            $post->post_type = 'xml_wp_query_item';
            $post->post_status = 'publish';
            $post->post_name = strval($item->slug);
            $post->post_date = strval($item->post_date);
            $post->ID = strval($item->ID);

            if (property_exists($item, 'post_title')) {
                $post->post_title = $item->post_title;
            } else {
                $post->post_title = xml_wp_query_random_string();
            }

            // Set post_content with a json string of the item to make other properties available and filterable
            $post->post_content = json_encode($item);
            array_push($posts, $post);
        }

        if (array_key_exists('fields_unique', $this->query_vars) && boolval($this->query_vars['fields_unique'])) {
            $posts = array_unique($posts);
        }

        $this->order_posts($posts);

        $this->found_posts = count($posts);

        $this->paginate($posts);

        return $posts;
    }

    /**
     * Check if the specified item meets the ID requirements.
     *
     * @param $item
     * @return bool
     */
    private function compare_id($item) {
        if (array_key_exists('p', $this->query_vars) && strval($item->ID) !== trim(strval($this->query_vars['p']))) {
            return false;
        }

        if (array_key_exists('post__in', $this->query_vars)) {
            $ids = $this->query_vars['post__in'];
            if (is_string($ids)) {
                $ids = explode(',', $ids);
            }
            array_walk($ids, function (&$id) {
                return trim(strval($id));
            });
            if (!in_array(strval($item->ID), $ids)) {
                return false;
            }
        }

        if (array_key_exists('post__not_in', $this->query_vars)) {
            $ids = $this->query_vars['post__not_in'];
            if (is_string($ids)) {
                $ids = explode(',', $ids);
            }
            array_walk($ids, function (&$id) {
                return trim(strval($id));
            });
            if (in_array(strval($item->ID), $ids)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Check if the specified item meets the date requirements.
     *
     * @param $item
     * @return bool
     */
    private function compare_date($item) {
        // Get the date
        $date_string = strval($item->post_date);
        $timestamp = strtotime($date_string);

        // Check if the date meets the date filters
        if (array_key_exists('year', $this->query_vars) && (!$date_string || date('Y', $timestamp) !== absint($this->query_vars['year']))) {
            return false;
        }
        if (array_key_exists('monthnum', $this->query_vars) && (!$date_string || date('n', $timestamp) !== absint($this->query_vars['monthnum']))) {
            return false;
        }
        if (array_key_exists('day', $this->query_vars) && (!$date_string || date('j', $timestamp) !== absint($this->query_vars['day']))) {
            return false;
        }
        if (array_key_exists('date_query', $this->query_vars)) {
            if (!$date_string) {
                return false;
            }
            $date_query = $this->query_vars['date_query'];
            krsort($date_query);
            $query_date = implode('-', array_intersect_key($date_query, array_flip(['year', 'month', 'day'])));
            if (!strstr($date_string, $query_date)) {
                return false;
            }
            if (array_key_exists('after', $date_query)) {
                $after = $date_query['after'];
                if (is_array($after)) {
                    krsort($after);
                    $after = implode('-', $after);
                }
                if ($timestamp < strtotime($after)) {
                    return false;
                }
            }
            if (array_key_exists('before', $date_query)) {
                $before = $date_query['before'];
                if (is_array($before)) {
                    krsort($before);
                    $before = implode('-', $before);
                }
                if ($timestamp > strtotime($before)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Check if the specified item meets the slug requirements.
     *
     * @param $item
     * @return bool
     */
    private function compare_slug($item) {
        if (array_key_exists('name', $this->query_vars) && strval($item->slug) !== trim($this->query_vars['name'])) {
            return false;
        }

        if (array_key_exists('post_name__in', $this->query_vars)) {
            $names = $this->query_vars['post_name__in'];
            if (is_string($names)) {
                $names = explode(',', $names);
            }
            array_walk($names, function (&$name) {
                return trim($name);
            });
            if (!in_array(strval($item->slug), $names)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Check if the specified item meets the meta requirements.
     *
     * @param $item
     * @return bool
     * @throws Exception
     */
    private function compare_meta($item)
    {
        // Check if the other properties meet the meta filters
        if (array_key_exists('meta_query', $this->query_vars)) {
            if (!$this->compare_meta_query($this->query_vars['meta_query'], $item)) {
                return false;
            }
        }
        if (array_key_exists('meta_key', $this->query_vars) || array_key_exists('key', $this->query_vars)) {
            $meta = [];
            if (array_key_exists('meta_key', $this->query_vars)) {
                $meta['key'] = $this->query_vars['meta_key'];
            } else {
                throw new Exception(__('XML WP_Query - No meta_key was given', 'xml-wp-query'));
            }

            if (array_key_exists('meta_value', $this->query_vars)) {
                $meta['value'] = $this->query_vars['meta_value'];
            } else {
                throw new Exception(__('XML WP_Query - No meta_value was given', 'xml-wp-query'));
            }

            if (array_key_exists('meta_compare', $this->query_vars)) {
                $meta['compare'] = $this->query_vars['meta_compare'];
            }

            if (!$this->compare_meta_query($meta, $item)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Parse meta query and compare the specified item.
     *
     * @param $query array {
     *      Array or string of Query parameters.
     * @type string    key         Custom field key.
     * @type mixed     value       Custom field value.
     * @type string    comparator  Comparison operator to test the 'meta_value'. Defaults to '='.
     * @type string    relation    Optional. Wether or not all the comparisons have to be true. Defaults to 'AND'.
     * }
     * @param object $item the object containing the properties to compare the values to.
     * @param string $relation Optional. Wether or not all the comparisons have to be true. Defaults to 'AND'.
     * @return bool
     */
    private function compare_meta_query($query, $item, $relation = 'AND')
    {
        // If relation is set, pull it from the array and compare by running this function again
        if (array_key_exists('relation', $query)) {
            $new_query = $query;
            $relation = $query['relation'];
            unset($new_query['relation']);
            return $this->compare_meta_query($new_query, $item, $relation);
        }

        // Loop the query to compare the property with the value
        // If relation is 'AND' and the comparison equals false, return false immediately
        // Else, add the comparison to the results array
        $results = [];
        foreach ($query as $meta) {
            if (!array_key_exists('compare', $meta)) {
                $meta['compare'] = '=';
            }

            $expected = null;
            if (array_key_exists('type', $meta) && in_array($meta['type'], ['ARRAY_COUNT', 'CHILD_COUNT'])) {
                if (property_exists($item, $meta['key'])) {
                    $value = $item->{$meta['key']};
                } else {
                    $value = $item->xpath($meta['key']);
                }
            } else if (property_exists($item, $meta['key'])) {
                $value = strval($item->{$meta['key']});
            } else {
                $value = $item->xpath($meta['key'] . '/text()');
                if (is_array($value)) {
                    $value = strval($value[0]);
                }
            }
            if (array_key_exists('type', $meta)) {
                // default supported by WP_Query: 'BINARY', 'CHAR', 'DATE', 'DATETIME', 'DECIMAL', 'SIGNED', 'TIME', 'UNSIGNED'.
                switch ($meta['type']) {
                    case 'SIGNED':
                    case 'UNSIGNED':
                        $value = (int)preg_replace('/[^0-9]/', '', $value);
                        $expected = (int)$meta['value'];
                        break;
                    case 'DUTCH_NUMBER':
                        $value = (float)str_replace(',', '.', str_replace('.', '', $value));
                        $expected = (float)$meta['value'];
                        break;
                    case 'NUMERIC':
                    case 'DECIMAL':
                        $value = (float)$value;
                        $expected = (float)$meta['value'];
                        break;
                    case 'DATE':
                    case 'DATETIME':
                    case 'TIME':
                        $value = strtotime(strval($value));
                        $expected = strtotime($meta['value']);
                        break;
                    case 'ARRAY_COUNT':
                        $value = count($value);
                        $expected = (int)$meta['value'];
                        break;
                    case 'CHILD_COUNT':
                        if (property_exists($value, 'childNodes')) {
                            $value = count($value->childNodes);
                        } else {
                            $value = count((array)$value);
                        }
                        $expected = (int)$meta['value'];
                        break;
                }
            }
            if (is_null($value)) {
                $value = '';
            }
            if (is_null($expected)) {
                $expected = $meta['value'];
            }

            switch ($meta['compare']) {
                case "SANITIZED":
                    $result = sanitize_title($value) == sanitize_title($expected);
                    break;
                case "LIKE":
                    $result = strstr($value, $expected);
                    break;
                case "NOT LIKE":
                    $result = !strstr($value, $expected);
                    break;
                case "REGEXP":
                    $result = preg_match($expected, $value);
                    break;
                case "NOT REGEXP":
                    $result = !preg_match($expected, $value);
                    break;
                case "!=":
                    $result = $value != $expected;
                    break;
                case ">=":
                    $result = $value >= $expected;
                    break;
                case "<=":
                    $result = $value <= $expected;
                    break;
                case ">":
                    $result = $value > $expected;
                    break;
                case "<":
                    $result = $value < $expected;
                    break;
                default:
                    $result = $value == $expected;
                    break;
            }

            if ($relation === 'AND' && !$result) {
                return false;
            } else {
                array_push($results, $result);
            }
        }

        // Check if any of the comparisons equals true
        $result = array_filter($results, function ($result) {
            return $result;
        });
        return count($result) > 0;
    }

    /**
     * Fetch specified fields instead of converting the item to a WP_Post object.
     *
     * @param array $posts
     * @param SimpleXMLElement $item
     */
    public function pluck(&$posts, &$item) {
        switch ($this->query_vars['fields']) {
            case 'ids':
                array_push($posts, strval($item->ID));
                break;
            case 'id=>title':
                if (property_exists($item, 'post_title')) {
                    $posts[strval($item->ID)] = $item->post_title;
                }
                break;
            case 'id=>slug':
                if (property_exists($item, 'slug')) {
                    $posts[strval($item->ID)] = $item->slug;
                }
                break;
            default:
                if (property_exists($item, $this->query_vars['fields'])) {
                    array_push($posts, strval($item->{$this->query_vars['fields']}));
                } else {
                    array_push($posts, $item->xpath($this->query_vars['fields']));
                }
                break;
        }
    }

    /**
     * Order the posts array.
     *
     * @param $posts
     */
    private function order_posts(&$posts) {
        $order = 'asc';
        $order_by = 'ID';

        // If a custom order was set, order the posts array accordingly
        if ($this->query_vars && (array_key_exists('orderby', $this->query_vars) || array_key_exists('order', $this->query_vars))) {
            if (array_key_exists('order', $this->query_vars) && strtolower($this->query_vars['order']) === 'desc') {
                $order = 'desc';
            }

            if (array_key_exists('orderby', $this->query_vars)) {
                $order_by = $this->query_vars['orderby'];
            }
        }

        if ($order_by === 'rand') {
            shuffle($posts);
        } else if (strstr($order_by, '->')) {
            $order_by = explode('->', $order_by);
            usort($posts, function ($a, $b) use ($order, $order_by) {
                $value_a = $a;
                $value_b = $b;
                foreach ($order_by as $property) {
                    if ($property === 'post_content') {
                        $value_a = json_decode($value_a->$property);
                        $value_b = json_decode($value_b->$property);
                    } else {
                        $value_a = $value_a->$property;
                        $value_b = $value_b->$property;
                    }
                }
                return xml_wp_query_compare($value_a, $value_b, $order);
            });
        } else if (is_object($posts[0])) {
            usort($posts, function ($a, $b) use ($order, $order_by) {
                $value_a = $a->$order_by;
                $value_b = $b->$order_by;
                return xml_wp_query_compare($value_a, $value_b, $order);
            });
        } else {
            usort($posts, function ($a, $b) use ($order) {
                return xml_wp_query_compare($a, $b, $order);
            });
        }
    }

    /**
     * Filter the posts to return only a subset if required.
     *
     * @param $posts
     */
    private function paginate(&$posts) {
        // Define pagination variables and unset their query_vars to prevent unnecessary compares
        $posts_per_page = get_option('posts_per_page');
        if ($this->query_vars && array_key_exists('posts_per_page', $this->query_vars)) {
            $posts_per_page = $this->query_vars['posts_per_page'];
        }

        $noPaging = false;
        if ($this->query_vars && array_key_exists('nopaging', $this->query_vars)) {
            $noPaging = $this->query_vars['nopaging'];
        }

        if ($posts_per_page > 0 && !$noPaging) {
            $page = 1;
            if ($this->query_vars && array_key_exists('paged', $this->query_vars)) {
                $page = $this->query_vars['paged'] > 1 ? $this->query_vars['paged'] : 1;
            }

            $offset = $posts_per_page * ($page - 1);
            if ($this->query_vars && array_key_exists('offset', $this->query_vars)) {
                $offset = $this->query_vars['offset'];
            }
        }

        // If pagination is enabled and not unlimited ($posts_per_page -1), return a subset of the posts array
        if ($posts_per_page > 0 && !$noPaging) {
            $this->max_num_pages = ceil($this->found_posts / $posts_per_page);
            $posts = array_slice($posts, $offset, $posts_per_page);
        }

        $this->post_count = count($posts);
        $this->posts = $posts;
    }
}